import React from 'react'
import DefaultView from 'views/DefaultView'
import './UserTable.scss'
import { fetchUsers } from 'modules/user_data/actions'
import { fetchUserTableRoute, toggleUserLocks } from '../actions'
import { UserLockType, UserLockState } from 'modules/user_data/objects/UserLock'
import ToggleConfirmModal from './ToggleConfirmModal'
import ToggleProgressModal from './ToggleProgressModal'
import UserLockCell from './UserLockCell'
import UserLockBars from './UserLockBars'
import UserTableHeader from './UserTableHeader'
import TableSorter, { TableSortType } from '../objects/TableSorter'
import {
  Grid, Row, Col,
  Glyphicon,
  Table,
  Button, ButtonGroup,
  Well,
  FormGroup, FormControl, InputGroup
} from 'react-bootstrap'

export default class UserTable extends DefaultView {

  constructor (props) {
    super(props)
    this.state = {
      selectedUserLocks: [],
      showToggleConfirmModal: false,
      showToggleProgressModal: false,
      sorter: new TableSorter('user_type'),
      filterValue: '',
      isLiveMode: true
    }
    this.tick = this.tick.bind(this)
    this.handleToggleSort = this.handleToggleSort.bind(this)
    this.handleCloseToggleModal = this.handleCloseToggleModal.bind(this)
    this.handleSelectUserLock = this.handleSelectUserLock.bind(this)
    this.toggleSelectedUserLocks = this.toggleSelectedUserLocks.bind(this)
    this.handleFilterValue = this.handleFilterValue.bind(this)
  }

  documentTitle () {
    return 'Heimdall'
  }

  headerTitle () {
    return 'Heimdall'
  }

  componentDidMount () {
    this.props.dispatch(fetchUserTableRoute())
    this.timer = setInterval(this.tick, 5000)
  }

  componentWillUnmount () {
    if (this.state.isLiveMode) {
      clearInterval(this.timer)
    }
  }

  componentDidUpdate (prevProps, prevState) {
    if (prevState.isLiveMode === false && this.state.isLiveMode === true) {
      this.timer = setInterval(this.tick, 1000)
    } else if (prevState.isLiveMode === true && this.state.isLiveMode === false) {
      clearInterval(this.timer)
    }
  }

  tick () {
    if (this.props.loading) {
      return
    }
    this.props.dispatch(fetchUsers())
  }

  setStateWithSelectedUserLocks (selectedUserLocks) {
    if (selectedUserLocks.length === 0 && this.state.isLiveMode === false) {
      console.log('Enabling live mode!')
      this.setState({ selectedUserLocks, isLiveMode: true })
    } else if (selectedUserLocks.length > 0 && this.state.isLiveMode === true) {
      console.log('Disabling live mode!')
      this.setState({ selectedUserLocks, isLiveMode: false })
    } else {
      this.setState({ selectedUserLocks })
    }
  }

  handleToggleSort (sorter) {
    this.setState({ sorter })
  }

  handleFilterValue (e) {
    this.setState({ filterValue: e.target.value.replace(/\s+/g, '') })
  }

  handleCloseToggleModal () {
    this.setState({ showToggleProgressModal: false })
    this.setStateWithSelectedUserLocks([])
    this.props.dispatch(fetchUsers())
  }

  handleSelectUserLock (user, lockType) {
    const lock = user.locks[lockType]
    lock.selected = !lock.selected
    let selectedUserLocks = this.state.selectedUserLocks
    let idx = 0
    for (idx = 0; idx < selectedUserLocks.length; idx++) {
      const pair = selectedUserLocks[idx]
      if (user.id === pair[0]) {
        if (lockType === pair[1]) {
          break
        }
      }
    }
    if (idx !== selectedUserLocks.length) {
      selectedUserLocks.splice(idx, 1)
    } else {
      selectedUserLocks.push([user.id, lockType])
    }
    this.setStateWithSelectedUserLocks(selectedUserLocks)
  }

  toggleSelectedUserLocks () {
    this.setState({ showToggleProgressModal: true })
    let toggles = []
    this.state.selectedUserLocks.forEach((pair) => {
      const user = this.props.users[pair[0]]
      const lockType = pair[1]
      const userType = this.props.userTypes[user.userTypeId]
      toggles.push([user, userType, lockType])
    })
    this.props.dispatch(toggleUserLocks(toggles))
  }

  sortUsers (userIds) {
    return userIds.sort((userIdA, userIdB) => {
      const sortById = this.state.sorter.id
      if (sortById === null) {
        return 0
      }
      const userA = this.props.users[userIdA]
      const userB = this.props.users[userIdB]
      let comparison = 0
      if (sortById === 'user_type') {
        const typeNameA = userA.getUserType(this.props.userTypes).name.toLowerCase()
        const typeNameB = userB.getUserType(this.props.userTypes).name.toLowerCase()
        if (typeNameA > typeNameB) {
          comparison = 1
        } else if (typeNameA < typeNameB) {
          comparison = -1
        }
      } else if (sortById === 'user') {
        const nameA = userA.name.toLowerCase()
        const nameB = userB.name.toLowerCase()
        if (nameA > nameB) {
          comparison = 1
        } else if (nameA < nameB) {
          comparison = -1
        }
      } else {
        const lockType = (sortById === 'lock_prod') ? UserLockType.PROD : UserLockType.STAGING
        let lockA = userA.locks[lockType].state
        let lockB = userB.locks[lockType].state
        if (lockA === UserLockState.LOCKED && lockB === UserLockState.UNLOCKED) {
          comparison = 1
        } else if (lockA === UserLockState.UNLOCKED && lockB === UserLockState.LOCKED) {
          comparison = -1
        }
      }
      return (this.state.sorter.type === TableSortType.ASCENDING) ? comparison : (comparison * -1)
    })
  }

  filterUsers (userIds) {
    const filterValue = this.state.filterValue.toLowerCase()
    if (filterValue === '') {
      return userIds
    }
    let filteredUsers = []
    userIds.forEach((userId) => {
      const user = this.props.users[userId]
      const stringsToVerify = [
        user.name,
        user.getUserType(this.props.userTypes).name
      ]
      for (let str of stringsToVerify) {
        if (str.toLowerCase().includes(filterValue)) {
          filteredUsers.push(userId)
          break
        }
      }
      if (filterValue === 'locked' || filterValue === 'unlocked') {
        const lockState = (filterValue === 'locked') ? UserLockState.LOCKED : UserLockState.UNLOCKED
        if (user.locks[UserLockType.PROD].state === lockState || user.locks[UserLockType.STAGING].state === lockState) {
          filteredUsers.push(userId)
        }
      } else if (filterValue === 'selected') {
        if (user.locks[UserLockType.PROD].selected || user.locks[UserLockType.STAGING].selected) {
          filteredUsers.push(userId)
        }
      }
    })
    return filteredUsers
  }

  renderContent () {
    let tableRows = []
    let lockStatusTotals = {}
    lockStatusTotals[UserLockType.PROD] = {}
    lockStatusTotals[UserLockType.PROD][UserLockState.UNLOCKED] = 0
    lockStatusTotals[UserLockType.PROD][UserLockState.LOCKED] = 0
    lockStatusTotals[UserLockType.STAGING] = {}
    lockStatusTotals[UserLockType.STAGING][UserLockState.UNLOCKED] = 0
    lockStatusTotals[UserLockType.STAGING][UserLockState.LOCKED] = 0
    let userIds = this.sortUsers(Object.keys(this.props.users))
    let numTotalUsers = userIds.length
    userIds = this.filterUsers(userIds)
    userIds.forEach((userId) => {
      const user = this.props.users[userId]
      const userType = user.getUserType(this.props.userTypes)
      lockStatusTotals[UserLockType.PROD][user.locks[UserLockType.PROD].state] += 1
      lockStatusTotals[UserLockType.STAGING][user.locks[UserLockType.STAGING].state] += 1
      tableRows.push(
        <tr key={user.id}>
          <td>{userType.name}</td>
          <td>{user.name}</td>
          <td>
            <UserLockCell
              user={user}
              lockType={UserLockType.PROD}
              onSelect={this.handleSelectUserLock}
              selectedUserLocks={this.state.selectedUserLocks}
            />
          </td>
          <td>
            <UserLockCell
              user={user}
              lockType={UserLockType.STAGING}
              onSelect={this.handleSelectUserLock}
              selectedUserLocks={this.state.selectedUserLocks}
            />
          </td>
        </tr>
      )
    })

    let statusStr = ''
    let statusStyle = 'status_str'
    if (this.state.selectedUserLocks.length > 0) {
      statusStr = `Selected ${this.state.selectedUserLocks.length} locks. ` + statusStr
    }
    let filterStr = 'Use "selected", "locked", or "unlocked" to filter by locks'
    let filterStyle = 'status_str'
    if (userIds.length === 0) {
      filterStyle += ' status_str_danger'
      filterStr = 'No users found that match filter'
    } else if (userIds.length !== numTotalUsers) {
      filterStr = `Showing ${userIds.length} of ${numTotalUsers} users`
    }
    statusStr = <div className={statusStyle}>{statusStr}</div>
    filterStr = <div className={filterStyle}><i>{filterStr}</i></div>

    const liveModeStr = this.state.isLiveMode ? 'Enabled' : 'Disabled'
    const liveModeGlyph = this.state.isLiveMode ? 'eye-open' : 'eye-close'

    return (
      <Grid style={{ marginTop: '40px' }}>
        <Row>
          <Col sm={12}>
            <Well>
              <table style={{ width: '100%' }}>
                <tbody>
                  <tr>
                    <td colSpan={3}>
                      <UserLockBars lockStatusTotals={lockStatusTotals} />
                    </td>
                  </tr>
                  <tr>
                    <td style={{ width: '99%', paddingRight: '20px' }}>
                      <form>
                        <FormGroup
                          controlId='formBasicText'
                          style={{ marginBottom: '0px' }}
                        >
                          <InputGroup>
                            <InputGroup.Addon>Filter Table:</InputGroup.Addon>
                            <FormControl
                              type='text'
                              value={this.state.filterValue}
                              placeholder='Enter text'
                              onChange={this.handleFilterValue}
                            />
                          </InputGroup>
                        </FormGroup>
                      </form>
                    </td>
                    <td style={{ whiteSpace: 'nowrap', paddingRight: '20px' }}>
                      <div>
                        <Glyphicon glyph={liveModeGlyph} /> Live Mode {liveModeStr}
                      </div>
                    </td>
                    <td style={{ whiteSpace: 'nowrap', textAlign: 'right' }}>
                      <ButtonGroup>
                        <Button
                          onClick={() => { this.setState({ showToggleConfirmModal: true }) }}
                          disabled={this.state.selectedUserLocks.length === 0}
                        >
                          Toggle Selected <Glyphicon glyph='lock' />
                        </Button>
                      </ButtonGroup>
                    </td>
                  </tr>
                  <tr>
                    <td colSpan={3}>
                      <table style={{ width: '100%' }}>
                        <tbody>
                          <tr>
                            <td style={{ whiteSpace: 'nowrap', paddingTop: '10px' }}>
                              {filterStr}
                            </td>
                            <td style={{ whiteSpace: 'nowrap', textAlign: 'right', paddingTop: '10px' }}>
                              {statusStr}
                            </td>
                          </tr>
                        </tbody>
                      </table>
                    </td>
                  </tr>
                </tbody>
              </table>
            </Well>
          </Col>
        </Row>
        <Row>
          <Col sm={12}>
            <Table striped>
              <thead>
                <tr>
                  <UserTableHeader
                    id={'user_type'}
                    sorter={this.state.sorter}
                    onToggleSort={this.handleToggleSort}
                  >
                    User
                  </UserTableHeader>
                  <UserTableHeader
                    id={'user'}
                    sorter={this.state.sorter}
                    onToggleSort={this.handleToggleSort}
                  >
                    Account
                  </UserTableHeader>
                  <UserTableHeader
                    id={'lock_prod'}
                    sorter={this.state.sorter}
                    onToggleSort={this.handleToggleSort}
                  >
                    <Glyphicon glyph='lock' /> Production
                  </UserTableHeader>
                  <UserTableHeader
                    id={'lock_staging'}
                    sorter={this.state.sorter}
                    onToggleSort={this.handleToggleSort}
                  >
                    <Glyphicon glyph='lock' /> Staging
                  </UserTableHeader>
                </tr>
              </thead>
              <tbody>
                {tableRows}
              </tbody>
            </Table>
          </Col>
        </Row>
        <ToggleConfirmModal
          isVisible={this.state.showToggleConfirmModal}
          onConfirm={this.toggleSelectedUserLocks}
          onCancel={() => { this.setState({ showToggleConfirmModal: false }) }}
          users={this.props.users}
          selectedUserLocks={this.state.selectedUserLocks}
        />
        <ToggleProgressModal
          isVisible={this.state.showToggleProgressModal}
          onConfirm={this.handleCloseToggleModal}
          users={this.props.users}
          selectedUserLocks={this.state.selectedUserLocks}
          isToggling={this.props.isTogglingUserLocks}
        />
      </Grid>
    )
  }
}
