/**
 * Class with a collection of static functions for performing various api calls.
 */

import { Enum } from 'enumify'

class ApiRequestType extends Enum {}
ApiRequestType.initEnum([
  'GET',
  'POST',
  'PUT'
])

export class ApiRequestError extends Error {
  constructor (status, message) {
    super(message)
    this.status = status
  }
}

export default class ApiUtil {

  static request (type, url, json) {
    return new Promise(function (resolve, reject) {
      var req = new XMLHttpRequest()
      req.open(type.name, url)
      req.onload = function () {
        if (req.status === 200) {
          resolve(req.response)
        } else {
          reject(new ApiRequestError(req.status, JSON.parse(req.response).message))
        }
      }
      req.onerror = function () {
        reject(new ApiRequestError(-1, `Network Error: '${url}'`))
      }
      req.send(json)
    })
  }

  static getJson (url) {
    return ApiUtil.request(ApiRequestType.GET, url).then(JSON.parse)
  }

  static getUserTypes () {
    let url = ApiUtil.urlPrefix + 'types'
    return ApiUtil.getJson(url)
  }

  static getUsers () {
    let url = ApiUtil.urlPrefix + `users`
    return ApiUtil.getJson(url)
  }

  static putUserLock (userTypeName, userName, lockEnv, lockAction) {
    let url = ApiUtil.urlPrefix + `types/${userTypeName}/users/${userName}/${lockEnv}/${lockAction}`
    return ApiUtil.request(ApiRequestType.PUT, url).then(JSON.parse)
  }
}

ApiUtil.urlPrefix = 'https://api.heimdall.xarth.tv/'
// ApiUtil.urlPrefix = 'http://localhost:3000/'
// ApiUtil.urlPrefix = 'http://api-dev.heimdall.xarth.tv/'
