// This module creates a Heimdall UI Environment.
// Useful for a moduel so that you don't have to repeat for staging vs prod

variable "application_name" {}
variable "environment_name" {}
variable "elb_sg" {}
variable "ec2_sg" {}
variable "r53_name" {}
# https://docs.aws.amazon.com/elasticbeanstalk/latest/platforms/platforms-supported.html#platforms-supported.docker
variable "solution_stack" {
  default = "64bit Amazon Linux 2018.03 v2.12.14 running Docker 18.06.1-ce"
}

variable "asg_min_size" {
  default = "1"
}

variable "asg_max_size" {
  default = "1"
}

// Used for accessing VPC / Subnets
data "terraform_remote_state" "parent_account_vpc" {
  backend = "s3"
  config = {
    profile = "twitch-cape-qe-aws"
    bucket  = "tf-state-qe"
    key     = "tfstate/qe/terraform/vpc"
    region  = "us-west-2"
  }
}

// Create a template for launching the beanstalk environment
resource "aws_elastic_beanstalk_configuration_template" "heimdall" {
  name                = "${var.environment_name}-template"
  application         = var.application_name
  solution_stack_name = var.solution_stack

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name = "InstanceType"
    value = "t3.micro"
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name = "MinSize"
    value = var.asg_min_size
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name = "MaxSize"
    value = var.asg_max_size
  }

  setting {
    namespace = "aws:ec2:vpc"
    name = "VPCId"
    value = data.terraform_remote_state.parent_account_vpc.outputs.vpc_id
  }

  setting {
    namespace = "aws:ec2:vpc"
    name = "Subnets"
    value = data.terraform_remote_state.parent_account_vpc.outputs.private_a
  }

  setting {
    namespace = "aws:ec2:vpc"
    name = "ELBSubnets"
    value = data.terraform_remote_state.parent_account_vpc.outputs.private_a
  }

  setting {
    namespace = "aws:ec2:vpc"
    name = "ELBScheme"
    value = "internal"
  }

  setting {
    namespace = "aws:elb:loadbalancer"
    name = "SecurityGroups"
    value = var.elb_sg
  }

  setting {
    namespace = "aws:elb:loadbalancer"
    name = "ManagedSecurityGroup"
    value = var.elb_sg
  }

  ############
  # Port 443 #
  ############
  setting {
    namespace = "aws:elb:listener:443"
    name = "ListenerProtocol"
    value = "HTTPS"
  }

  setting {
    namespace = "aws:elb:listener:443"
    name = "InstancePort"
    value = "443"
  }

  setting {
    namespace = "aws:elb:listener:443"
    name = "InstanceProtocol"
    value = "HTTPS"
  }

  setting {
    namespace = "aws:elb:listener:443"
    name = "SSLCertificateId"
    value = aws_acm_certificate.xarth_cert.arn
  }

  ##########
  # Others #
  ##########
  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name = "SecurityGroups"
    value = var.ec2_sg
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name = "ENVIRONMENT_NAME"
    value = var.environment_name
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name = "IamInstanceProfile"
    value = aws_iam_instance_profile.heimdall-ui.arn
  }
}

// Spin up the environment using the above template
resource "aws_elastic_beanstalk_environment" "heimdall" {
  name                = "${var.application_name}-${var.environment_name}"
  application         = var.application_name
  template_name       = aws_elastic_beanstalk_configuration_template.heimdall.name
}

################################
###### Instance Profiles #######
################################

resource "aws_iam_instance_profile" "heimdall-ui" {
  name = "${var.application_name}-${var.environment_name}-ip"
  role = aws_iam_role.heimdall-ui.name
}

resource "aws_iam_role" "heimdall-ui" {
  name = "${var.application_name}-${var.environment_name}-role"
  assume_role_policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "sts:AssumeRole",
            "Principal": {
               "Service": "ec2.amazonaws.com"
            },
            "Effect": "Allow",
            "Sid": ""
        }
    ]
}
EOF
}

resource "aws_iam_policy" "metrics_monitoring" {
  name        = "MetricsMonitoring-${var.application_name}-${var.environment_name}"
  description = "Used to post and get metrics for EC2 instances to Cloudwatch for ${var.application_name} ${var.environment_name}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1498371131251",
      "Action": [
        "cloudwatch:GetMetricStatistics",
        "cloudwatch:ListMetrics",
        "cloudwatch:PutMetricData"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "Stmt1498371168662",
      "Action": [
        "ec2:DescribeTags"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_iam_policy_attachment" "metrics_monitoring" {
  name = "MetricsMonitoring-${var.application_name}-${var.environment_name}"
  roles  = [aws_iam_role.heimdall-ui.name]
  policy_arn = aws_iam_policy.metrics_monitoring.arn
}

resource "aws_iam_policy_attachment" "beanstalk_web_tier" {
  name = "Beanstalk Web Tier"
  roles  = [aws_iam_role.heimdall-ui.name]
  policy_arn = "arn:aws:iam::aws:policy/AWSElasticBeanstalkWebTier"
}

# Allow AWS SSM Access
resource "aws_iam_role_policy_attachment" "ssm" {
  role       = aws_iam_role.heimdall-ui.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore"
}

output "cname" {
  value = aws_elastic_beanstalk_environment.heimdall.cname
}
