# app/controllers/locks_controller.rb
class LocksController < ApplicationController
  # In minutes the length of time a lock good for unless another number is specified
  #  in the ttl query parameter
  DEFAULT_TTL = 60

  # GET /locks
  def index
    json_response(Lock.all.reload)
  end

  # GET /types/:type_name/users/:user_name/locks
  def index_user
    response_or_404(get_user&.locks&.reload, 'User not found')
  end

  # GET /types/:type_name/users/:user_name/:env
  def show
    response_or_404(get_user&.lock(environment)&.reload, 'User not found')
  end

  # PUT /types/:type_name/users/:user_name/:env/lock
  def lock
    user = get_user
    return not_found_response('User not found') if user.blank?
    return json_response(user) if acquire_lock(user)

    message = { message: 'Lock failed: User already locked' }
    json_response(message, :unprocessable_entity)
  end

  # PUT /types/:type_name/users/:user_name/:env/unlock
  def unlock
    user = get_user
    return not_found_response('User not found') if user.blank?
    return json_response(user) if release_lock(user)

    message = { message: 'Failed to Unlock User' }
    json_response(message, :unprocessable_entity)
  end

  # PUT /types/:type_name/available/:env
  # PUT /types/available/:env
  def available
    begin
      types = user_types
    rescue Exceptions::UserTypeNotFound => e
      message = { message: "Found invalid UserType(s): #{e.types}" }
      return json_response(message, :not_found)
    rescue Exceptions::UserTypeNotSpecified
      message = { message: "Must provide the key 'type_name' within the request body" }
      return json_response(message, :not_found)
    end

    locked_users = []
    unavailable_types = []

    types.each do |type|
      found_user = false
      type.users.each do |user|
        break locked_users << found_user = { UserType: type, User: user } if acquire_lock(user)
      end

      unavailable_types << type.name unless found_user
    end

    unless unavailable_types.empty?
      locked_users.each { |locked| release_lock(locked[:User]) }

      message = { message: "The following UserTypes were unavailable: #{unavailable_types.join(', ')}" }
      return json_response(message, :unprocessable_entity)
    end

    multi_part = locked_users.count > 1 || params[:version].to_s == '2'
    message = multi_part ? locked_users : locked_users.first[:User]
    json_response(message)
  end

  private

  ## Response helpers

  # Returns the 404 message wrapped as a json_response
  # @param message [String] the 404 message to wrap and return
  # @return [String, NilClass]
  def not_found_response(message)
    message = { message: message }
    json_response(message, :not_found)
  end

  # Returns the payload as a 200 json_response or a 404 json_response containing the missing message
  # @param payload [Object, NilClass] the payload we want to return to the user
  # @param missing_message [String] the 404 message if the payload isn't present
  # @return [String, NilClass] the response to return to the user
  def response_or_404(payload, missing_message)
    payload.present? ? json_response(payload) : not_found_response(missing_message)
  end

  ## Lock Helpers

  # Tries to acquire a lock on the user
  # @param user [User] the user we're locking
  # @return [Boolean] Was the lock successfully acquired
  def acquire_lock(user)
    lock = user.lock(environment)
    lock.with_lock do
      return false if lock.active?

      lock.state = true
      lock.build_id = params[:build_id]
      lock.thread_id = params[:thread_id]
      lock.test_id = params[:test_id]
      lock.job_name = params[:job_name]
      lock.expires_at = DateTime.now.utc.advance(minutes: params[:ttl] || DEFAULT_TTL)

      lock.save!
    end
  end

  # Tries to release a lock on the user.  This helper is more idempotent than the
  #   previous helper.  If it finds an inactive lock it still returns true.
  # @param user [User] the user we're unlocking
  # @return [Boolean] Was the lock already or successfully released
  def release_lock(user)
    # Thought: there's no validation that the caller is the same who created the lock
    #   Not a regression and as such this comment is here to say that we should
    #   consider detecting that and complaining by default

    lock = user.lock(environment)
    lock.with_lock do
      return true unless lock.active?

      lock.state = false
      lock.build_id = nil
      lock.thread_id = nil
      lock.test_id = nil
      lock.job_name = nil
      lock.expires_at = nil

      lock.save!
    end
  end

  ## Several Helpers to standardize parameter access

  # Normalizes the environment from the query parameters
  # @return [String] the normalized query parameter
  def environment
    env = params[:env] || 'prod'
    env == 'production' ? 'prod' : env.to_s
  end

  # Returns a the user matching the user type name (params[:type_name]) and username (params[:user_name])
  # @return [User, NilClass] returns either a user or nil if the user is not found
  def get_user
    User.joins(:user_type).where(user_types: { name: params[:type_name] }, username: params[:user_name]).take
  end

  # Parses the requested parameter and returns the correlated UserTypes
  # @param param [Symbol] the param to pull the user type names from
  # @raise [Exceptions::UserTypeNotSpecified] if the param is blank
  # @raise [Exceptions::UserTypeNotFound] if one or more of the user types was not found
  # @return [Array<UserType>] the matching UserTypes
  def user_types(param = :type_name)
    raise Exceptions::UserTypeNotSpecified if params[param].blank?

    requested_types = params[param].split(',')
    found_types = UserType.where(name: requested_types.to_set.to_a).to_a

    missing_types = (requested_types - found_types.collect(&:name)).to_a
    raise Exceptions::UserTypeNotFound.new(missing_types.join(', ')) unless missing_types.empty?

    requested_types.collect { |requested| found_types.find { |found| found.name == requested } }
  end
end
