require 'rails_helper'

RSpec.describe LocksController, type: :controller do

  let!(:user_type) { create(:user_type) }
  let!(:user) { create(:user, user_type_id: user_type.id) }

  describe 'Lock and Unlock' do
    let!(:env) { 'staging' }
    let!(:hour_from_now) { DateTime.now.utc.advance(minutes: 60) }

    let!(:type_1) { FactoryGirl.create(:user_type, name: 'type_1') }
    let!(:type_2) { FactoryGirl.create(:user_type, name: 'type_2') }

    # Duplicate Username Scenario based off bug QE-465
    let!(:user_1) do
      user = FactoryGirl.build(:user, username: 'qa_following', user_type_id: type_1.id)
      user.save(validate: false)
      user
    end
    let!(:user_2) do
      user = FactoryGirl.build(:user, username: 'qa_following', user_type_id: type_2.id)
      user.save!(validate: false)
      user
    end

    let!(:user_1_lock) { user_1.lock(env) }
    let!(:user_2_lock) { user_2.lock(env) }

    describe 'PUT unlock' do
      it 'retrieves the correct user of a usertype' do
        # Lock both users
        user_1_lock.update(state: true, expires_at: hour_from_now)
        user_2_lock.update(state: true, expires_at: hour_from_now)

        expect(user_1_lock.active?).to be_truthy
        expect(user_2_lock.active?).to be_truthy

        # Perform an unlock to the second user
        put :unlock, params: { type_name: type_2.name, user_name: user_2.username, env: env }
        expect(response).to have_http_status 200

        # Only user_2 should be unlocked
        expect(user_1_lock.active?).to be_truthy
        expect(user_2_lock.active?).to be_falsey
      end
    end

    describe 'PUT lock' do
      it 'expired locks do not block new locks' do
        # Create a lock and let it expire
        user_1_lock.update(state: true, expires_at: DateTime.now.utc)
        sleep 1

        expect(user_1_lock.active?).to be_falsey
        put :lock, params: { type_name: type_1.name, user_name: user_1.username, env: env }
        expect(response).to have_http_status 200

        # lock should now be active
        expect(user_1_lock.active?).to be_truthy
      end

      it 'active locks block new locks' do
        # Create a lock
        user_1_lock.update(state: true, expires_at: hour_from_now)

        expect(user_1_lock.active?).to be_truthy

        put :lock, params: { type_name: type_1.name, user_name: user_1.username, env: env }
        expect(response).to have_http_status 422

        expect(user_1_lock.active?).to be_truthy
      end
    end
  end

  describe 'PUT #available' do
    let!(:type_1) { FactoryGirl.create(:user_type, name: 'test_1') }
    let!(:user_1) { FactoryGirl.create(:user, user_type_id: type_1.id) }
    let!(:user_3) { FactoryGirl.create(:user, user_type_id: type_1.id) }

    let!(:type_2) { FactoryGirl.create(:user_type, name: 'test_2') }
    let!(:user_2) { FactoryGirl.create(:user, user_type_id: type_2.id) }

    it 'accepts 1 UserType' do
      put :available, params: { env: 'prod', type_name: type_1.name }
      expect(response).to have_http_status 200
    end

    it 'accepts multiple UserTypes' do
      put :available, params: { env: 'prod', type_name: "#{type_1.name},#{type_2.name}" }
      expect(response).to have_http_status 200
    end

    it 'accepts multiple UserTypes' do
      put :available, params: { env: 'prod', type_name: "#{type_1.name},#{type_1.name}" }
      expect(response).to have_http_status 200
    end

    it 'returns user accounts when all are available' do
      put :available, params: { env: 'prod', type_name: "#{type_1.name},#{type_2.name}" }
      expect(response).to have_http_status 200

      type_1_collection = parse_response_by_type(type_1)
      type_2_collection = parse_response_by_type(type_2)

      expect(type_1_collection).not_to be_nil
      expect(type_2_collection).not_to be_nil

      expect(type_1_collection.dig('UserType', 'name')).to eq(type_1.name)
      expect(type_1_collection.dig('User', 'username')).to eq(user_1.username)
      expect(type_2_collection.dig('UserType', 'name')).to eq(type_2.name)
      expect(type_2_collection.dig('User', 'username')).to eq(user_2.username)
    end

    it 'returns user accounts when all are available of same type' do
      put :available, params: { env: 'prod', type_name: "#{type_1.name},#{type_1.name}" }
      expect(response).to have_http_status 200
      expect(body_content.length).to eq(2)

      type_1_collection = body_content[0]
      type_2_collection = body_content[1]

      expect(type_1_collection).not_to be_nil
      expect(type_2_collection).not_to be_nil

      expect(type_1_collection.dig('UserType', 'name')).to eq(type_2_collection.dig('UserType', 'name'))
      expect(type_1_collection.dig('User', 'username')).not_to eq(type_2_collection.dig('User', 'username'))
    end

    it 'properly locks all UserTypes' do
      expect(user_locked?(user_1)).to be_falsey
      expect(user_locked?(user_2)).to be_falsey
      put :available, params: { env: 'prod', type_name: "#{type_1.name},#{type_2.name}" }
      expect(response).to have_http_status 200
      expect(user_locked?(user_1)).to be_truthy
      expect(user_locked?(user_2)).to be_truthy
    end

    it "doesn't lock accounts that were not passed" do
      put :available, params: { env: 'prod', type_name: type_1.name }
      expect(response).to have_http_status 200
      expect(user_locked?(user_1)).to be_truthy
      expect(user_locked?(user_2)).to be_falsey
    end

    it 'returns no account if 1/2 are unavailable and reverts lock actions' do
      # Only the first user is locked
      user_1.lock('prod').update(state: false)
      user_2.lock('prod').update(state: true, expires_at: DateTime.now.utc.advance(minutes: 60))

      expect(user_1.lock('prod').active?).to be_falsey
      expect(user_2.lock('prod').active?).to be_truthy

      put :available, params: { env: 'prod', type_name: "#{type_1.name},#{type_2.name}" }
      expect(response).to have_http_status 422

      # The state should be unchanged from above
      expect(user_1.lock('prod').active?).to be_falsey
      expect(user_2.lock('prod').active?).to be_truthy
    end

    it 'returns an error if the UserType provided is unavailable' do
      non_existent_type = 'foobar'
      put :available, params: { env: 'prod', type_name: non_existent_type }
      expect(response).to have_http_status 404
      expect(body_message).to eq("Found invalid UserType(s): #{non_existent_type}")
    end

    it 'gracefully errors if no user types are provided' do
      put :available, params: { env: 'prod' }
      expect(response).to have_http_status 404
      expect(body_message).to eq("Must provide the key 'type_name' within the request body")
    end

    it 'gracefully errors if users are not comma separated' do
      non_existent_type = "#{type_1.name}&#{type_2.name}"
      put :available, params: { env: 'prod', type_name: non_existent_type }
      expect(response).to have_http_status 404
      expect(body_message).to eq("Found invalid UserType(s): #{non_existent_type}")
    end

    describe 'responses' do

      describe 'v1' do
        it 'returns just the user if user supplies v1' do
          put :available, params: { env: 'prod', type_name: type_1.name }
          expect(response).to have_http_status 200
          expect(v1_response?(response.body)).to be_truthy
          expect(v2_response?(response.body)).to be_falsey
        end
      end

      describe 'v2' do
        it 'loads automatically when two users are passed' do
          put :available, params: { env: 'prod', type_name: "#{type_1.name},#{type_2.name}" } # Request both
          expect(response).to have_http_status 200
          expect(v1_response?(response.body)).to be_falsey
          expect(v2_response?(response.body)).to be_truthy
        end

        it 'loads when two users are passed even if version 1 is supplied' do
          put :available, params: { env: 'prod', type_name: "#{type_1.name},#{type_2.name}", version: '1' }
          expect(response).to have_http_status 200
          expect(v1_response?(response.body)).to be_falsey
          expect(v2_response?(response.body)).to be_truthy
        end

        it 'loads when a version parameter is passed' do
          put :available, params: { env: 'prod', type_name: type_1.name, version: '2' }
          expect(response).to have_http_status 200
          expect(v1_response?(response.body)).to be_falsey
          expect(v2_response?(response.body)).to be_truthy
        end
      end

      def v1_response?(body)
        body = JSON.parse(body)
        return body.has_key?('body') && body['body'].is_a?(Hash) && body['body'].has_key?('username')
      end

      def v2_response?(body)
        body = JSON.parse(body)
        return body.has_key?('body') && body['body'].is_a?(Array) && body['body'].first.has_key?('UserType')
      end
    end

    # Parses through the body JSON to return the UserType object
    # @param usertype [UserType] The UserType object looking for in the data
    # @return [Hash] if the usertype was found, the JSON response for that collection ({'UserType', 'User'})
    # @return [NilClass] if the usertype was not found
    def parse_response_by_type(usertype)
      body_content.each do |collection|
        return collection if collection['UserType']['id'] == usertype.id
      end

      return nil
    end

    # @return [Hash] The JSON parsed body content
    def body_content
      r = JSON.parse(response.body)
      return r.dig('body')
    end

    def body_message
      return body_content.dig('message')
    end

    # @param user [User] The user object to see if it's locked
    # @param env [String] The lock environment
    # @return [Boolean]
    def user_locked?(user, env = 'prod')
      return user.lock(env).active?
    end
  end

end
