# spec/requests/team_spec.rb
require 'rails_helper'

RSpec.describe 'Teams API', type: :request do
  # initialize test data
  let!(:teams) { create_list(:team, 10) }
  let(:teams_id) { teams.first.id }
  let(:teams_name) { teams.first.name }

  describe 'GET /teams/:team_name' do
    before { get "/teams/#{teams_name}" }

    context 'when team exists' do
      it 'returns status code 200' do
        expect(response).to have_http_status(200)
      end

      it 'returns the team id' do
        expect(json['body']['id']).to eq(teams_id)
      end
    end
  end

  describe 'GET /teams/:team_name' do
    before { get "/teams/Invalid_Name" }

    context 'when team does not exists' do
      it 'returns status code 404' do
        expect(response).to have_http_status(404)
      end
    end
  end

  describe 'POST /teams' do

    context 'when request attributes are valid' do
      name = 'New Test Team'
      before { post "/teams", params: { name: name  } }

      it 'returns status code 201' do
        expect(response).to have_http_status(201)
        expect(Team.exists?(name: name)).to be_truthy
      end

      it 'creates a team' do
        expect(json['body']['name']).to eq(name)
      end
    end

    context 'when the team already exists' do
      before { post "/teams", params: { name: teams_name } }

      it 'returns status code 422' do
        expect(response).to have_http_status(422)
      end

      it 'returns failure message ' do
        expect(response.body)
            .to match(/Validation failed: Name has already been taken/)
      end
    end

    context 'when request attributes are invalid' do
      before { post "/teams", params: {} }
      it 'returns status code 422' do
        expect(response).to have_http_status(422)
      end

      it 'returns a failure message' do
        expect(response.body).to match(/Validation failed: Name can't be blank/)
      end
    end
  end

  describe 'PUT /teams' do
    let(:valid_attributes) { { name: 'updated_test_team' } }
    before { put "/teams/#{teams_name}", params: valid_attributes }

    context 'when team exists' do
      it 'returns the team' do
        expect(json['body']['id']).to eq(teams_id)
      end

      it 'returns status code 200' do
        expect(response).to have_http_status(200)
      end

      it 'updates the team' do
        updated_team = Team.find(teams_id)
        expect(updated_team.name).to match(/updated_test_team/)
      end

    end
  end

end
