# spec/requests/user_types_spec.rb
require 'rails_helper'

RSpec.describe 'UserTypes API', type: :request do
  # initialize test data
  let!(:user_types) { create_list(:user_type, 10) }
  let(:type_name) { user_types.first.name }

  describe 'GET /types' do
    let(:endpoint) { '/types' }
    before { get endpoint }

    it 'returns user_types' do
      expect(json).not_to be_empty
      expect(json['body'].size).to eq(10)
    end

    it 'returns status code 200' do
      expect(response).to have_http_status(200)
    end

    describe 'request_uuid' do
      subject { json['request_uuid'] }

      context 'when not provided' do
        it do
          get endpoint
          should be_nil
        end
      end

      context 'when provided' do
        it do
          get endpoint + '?request_uuid=12345'
          should eq('12345')
        end
      end
    end
  end

  describe 'POST /types' do
    # valid payload
    let(:valid_attributes) { { name: 'new_user_type', description: 'my_description' } }

    context 'when the request is valid' do
      before { post '/types', params: valid_attributes }

      it 'creates a user_type' do
        expect(json['body']['name']).to eq('new_user_type')
      end

      it 'has a descrption' do
        expect(json['body']['description']).to eq('my_description')
      end

      it 'returns status code 201' do
        expect(response).to have_http_status(201)
      end
    end

    context 'when the request is invalid' do
      before { post '/types', params: {} }

      it 'returns status code 422' do
        expect(response).to have_http_status(422)
      end

      it 'returns a validation failure message' do
        expect(response.body)
          .to match(/Validation failed: Name can't be blank/)
      end
    end

    context 'when the user_type already exists' do
      before { post '/types', params: { name: type_name } }

      it 'returns status code 422' do
        expect(response).to have_http_status(422)
      end
      it 'returns a validation failure message' do
        expect(response.body)
          .to match(/Validation failed: UserType already exsits/)
      end
    end
  end

  describe 'PUT /types/:type_name' do
    let(:valid_attributes) { { name: 'updated_user_type' } }

    before { put "/types/#{type_name}", params: valid_attributes }

    context 'when the user_type exists' do

      it 'updates the user_type' do
        expect(json['body']['name']).to eq('updated_user_type')
      end

      it 'returns status code 200' do
        expect(response).to have_http_status(200)
      end
    end

    context 'when the user_type does not exist' do
      let(:type_name) { 'not_real_user_type' }

      it 'returns status code 404' do
        expect(response).to have_http_status(404)
      end

      it 'returns a not found message' do
        expect(response.body).to match(/Couldn't find UserType/)
      end
    end
  end
end
