# app/requests/users_spec.rb
require 'rails_helper'

RSpec.describe 'Users API' do
  # Initialize the test data
  let!(:user_type) { create(:user_type) }
  let!(:team) { create(:team) }
  let!(:users) { create_list(:user, 5, user_type_id: user_type.id, teams_id: team.id) }
  let(:type_name) { user_type.name }
  let(:teams_name) { team.name }
  let(:token) { team.api_key }
  let(:user_id) { users.first.id }
  let(:user_name) { users.first.username }

  describe 'GET /types/:type_name/users' do
    let(:endpoint) { "/types/#{type_name}/users" }
    before { get endpoint }

    context 'when user_type exists' do
      it 'returns status code 200' do
        expect(response).to have_http_status(200)
      end

      it 'returns all users' do
        expect(json['body'].size).to eq(5)
      end
    end

    context 'when user_type does not exist' do
      let(:type_name) { 'invalid_account_type' }

      it 'returns status code 404' do
        expect(response).to have_http_status(404)
      end

      it 'returns a not found message' do
        expect(response.body).to match(/Couldn't find UserType/)
      end
    end

    describe 'request_uuid' do
      subject { json['request_uuid'] }

      context 'when not provided' do
        it do
          get endpoint
          should be_nil
        end
      end

      context 'when provided' do
        it do
          get endpoint + '?request_uuid=12345'
          should eq('12345')
        end
      end
    end
  end

  describe 'GET /types/:type_name/users/:user_name' do
    before { get "/types/#{type_name}/users/#{user_name}" }

    context 'when user exists' do
      it 'returns status code 200' do
        expect(response).to have_http_status(200)
      end

      it 'returns the user' do
        expect(json['body']['id']).to eq(user_id)
      end
    end

    context 'when user does not exist' do
      let(:user_name) { 'invalid_account_type' }

      it 'returns status code 404' do
        expect(response).to have_http_status(404)
      end

      it 'returns a not found message' do
        expect(response.body).to match(/Couldn't find User/)
      end
    end
  end

  describe 'POST /types/:type_name/users' do
    let(:headers) { { Authorization: "Token token=#{token}"} }
    context 'when request attributes are valid' do
      let(:valid_attributes) { { username: 'new_username', locked_prod: false, locked_staging: false } }
      before { post "/types/#{type_name}/users", params: valid_attributes, headers: headers }

      it 'creates a user' do
        expect(json['body']['username']).to eq('new_username')
      end

      it 'returns status code 201' do
        expect(response).to have_http_status(201)
      end
    end

    context 'when an invalid request' do
      before { post "/types/#{type_name}/users", params: {teams_name: teams_name}, headers: headers }

      it 'returns status code 422' do
        expect(response).to have_http_status(422)
      end

      it 'returns a failure message' do
        expect(response.body).to match(/Validation failed: Username can't be blank/)
      end
    end

    context 'when the user already exists' do
      before { post "/types/#{type_name}/users", params: { username: user_name, teams_name: teams_name },
                    headers: headers }

      it 'returns status code 422' do
        expect(response).to have_http_status(422)
      end
      it 'returns a validation failure message' do
        expect(response.body)
          .to match(/Validation failed: User already exsits/)
      end
    end
  end

  describe 'PUT /types/:type_name/users/:user_name' do
    let(:valid_attributes) { { username: 'updated_username' } }
    let(:headers) { { Authorization: "Token token=#{token}"} }
    before { put "/types/#{type_name}/users/#{user_name}", params: valid_attributes, headers: headers }

    context 'when user exists' do
      it 'returns the user' do
        expect(json['body']['id']).to eq(user_id)
      end

      it 'returns status code 200' do
        expect(response).to have_http_status(200)
      end

      it 'updates the user' do
        updated_user = User.find(user_id)
        expect(updated_user.username).to match(/updated_username/)
      end
    end

    context 'when the user does not exist' do
      let(:user_name) { 'not_real_user' }

      it 'returns status code 404' do
        expect(response).to have_http_status(404)
      end

      it 'returns a not found message' do
        expect(response.body).to match(/Couldn't find User/)
      end
    end
  end

  describe 'DELETE /types/:type_name/users/:user_name' do
    let(:username) { users.last.username }
    let(:valid_attributes) { { username: 'updated_username' } }
    let(:headers) { { Authorization: "Token token=#{token}"} }
    before { delete "/types/#{type_name}/users/#{username}", params: valid_attributes, headers: headers }

    context 'when user exists' do
      it 'returns the user' do
        expect(json['body']['user']['id']).to eq(users.last.id)
      end

      it 'returns status code 200' do
        expect(response).to have_http_status(200)
      end
    end

    context 'when user does not exists' do
      before { delete "/types/#{type_name}/users/#{username}", params: valid_attributes, headers: headers }

      it 'returns status code 404' do
        expect(response).to have_http_status(404)
      end
    end
  end

end
