provider "aws" {
  region  = "us-west-2"
  profile = "twitch-cape-qe-aws"
  version = "~> 2.0"
}

resource "aws_elastic_beanstalk_application" "heimdall" {
  name        = "heimdall"
  description = "Service for managing Test Automation User Accounts"

  appversion_lifecycle {
    service_role          = data.aws_iam_role.beanstalk_service_role.arn
    max_count             = 100
    delete_source_from_s3 = true
  }
}

data "terraform_remote_state" "rds" {
  backend = "s3" # Capture the RDS Endpoints

  config = {
    profile = "twitch-cape-qe-aws"
    bucket  = "tf-state-qe"
    key     = "tfstate/qe/heimdall/rds"
    region  = "us-west-2"
  }
}

module "heimdall-production" {
  source               = "./modules"
  application_name     = aws_elastic_beanstalk_application.heimdall.name
  environment_name     = "production"
  r53_zone             = aws_route53_zone.heimdall.id
  dns_record_name      = "api"
  elb_sg               = aws_security_group.heimdall_elb.id
  ec2_sg               = aws_security_group.heimdall_ec2.id
  rds_db_name          = data.terraform_remote_state.rds.outputs.production_name
  rds_hostname         = data.terraform_remote_state.rds.outputs.production_address
  rds_port             = data.terraform_remote_state.rds.outputs.production_port
  sns_arn              = aws_sns_topic.heimdall.arn
  env_health_threshold = "20" // Degraded Level
  nexus_dns_name       = "nexus-Nexus-1RX7B0B2LLEB7-1733443262.us-west-2.elb.amazonaws.com"
}

output "heimdall_production_fqdn" {
  value = module.heimdall-production.fqdn
}

output "prod_acm_nexus_arn" {
  value = module.heimdall-production.acm_nexus_arn
}

module "heimdall-staging" {
  source           = "./modules"
  application_name = aws_elastic_beanstalk_application.heimdall.name
  environment_name = "staging"
  r53_zone         = aws_route53_zone.heimdall.id
  dns_record_name  = "api-dev"
  elb_sg           = aws_security_group.heimdall_elb.id
  ec2_sg           = aws_security_group.heimdall_ec2.id
  rds_db_name      = data.terraform_remote_state.rds.outputs.staging_name
  rds_hostname     = data.terraform_remote_state.rds.outputs.staging_address
  rds_port         = data.terraform_remote_state.rds.outputs.staging_port
  sns_arn          = aws_sns_topic.heimdall.arn
  alarms_enabled   = "false"
  nexus_dns_name   = "nexus-Nexus-1G44W9H79292M-616802512.us-west-2.elb.amazonaws.com"
}

output "heimdall_staging_fqdn" {
  value = module.heimdall-staging.fqdn
}

output "staging_acm_nexus_arn" {
  value = module.heimdall-staging.acm_nexus_arn
}
