variable "application_name" {}
variable "environment_name" {}
variable "elb_sg" {}
variable "ec2_sg" {}
variable "rds_db_name" {}
variable "rds_hostname" {}
variable "rds_port" {
  default = "3306"
}

# https://docs.aws.amazon.com/elasticbeanstalk/latest/platforms/platform-history-ruby.html
variable solution_stack {
  default = "64bit Amazon Linux 2018.03 v2.12.4 running Ruby 2.6 (Passenger Standalone)"
}

resource "aws_elastic_beanstalk_configuration_template" "heimdall" {
  name                = "${var.environment_name}-template"
  application         = var.application_name
  solution_stack_name = var.solution_stack

  setting {
    namespace = "aws:ec2:vpc"
    name = "VPCId"
    value = var.vpc_id
  }

  setting {
    namespace = "aws:ec2:vpc"
    name = "Subnets"
    value = join(", ", var.subnets)
  }

  setting {
    namespace = "aws:ec2:vpc"
    name = "ELBSubnets"
    value = join(", ", var.subnets)
  }

  setting {
    namespace = "aws:ec2:vpc"
    name = "ELBScheme"
    value = "internal"
  }

  setting {
    namespace = "aws:elb:loadbalancer"
    name = "SecurityGroups"
    value = var.elb_sg
  }

  setting {
    namespace = "aws:elb:loadbalancer"
    name = "ManagedSecurityGroup"
    value = var.elb_sg
  }

  ############
  # Port 443 #
  ############
  setting {
    namespace = "aws:elb:listener:443"
    name = "ListenerProtocol"
    value = "HTTPS"
  }

  setting {
    namespace = "aws:elb:listener:443"
    name = "InstancePort"
    value = "443"
  }

  setting {
    namespace = "aws:elb:listener:443"
    name = "InstanceProtocol"
    value = "HTTPS"
  }

  setting {
    namespace = "aws:elb:listener:443"
    name = "SSLCertificateId"
    value = aws_acm_certificate.xarth_cert_nexus.arn
  }

  ##########
  # Others #
  ##########

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name = "SecurityGroups"
    value = "${var.ec2_sg}, sg-771e780c"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name = "IamInstanceProfile"
    value = aws_iam_instance_profile.heimdall.arn
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name = "RDS_HOSTNAME"
    value = var.rds_hostname
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name = "RDS_PORT"
    value = var.rds_port
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name = "RDS_DB_NAME"
    value = "heimdall"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name = "RDS_USERNAME"
    value = "PLACEHOLDER"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name = "RDS_PASSWORD"
    value = "PLACEHOLDER"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name = "SECRET_KEY_BASE"
    value = "PLACEHOLDER"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name = "ENVIRONMENT_NAME"
    value = var.environment_name
  }

  ####################
  # Health Reporting #
  ####################
  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name      = "SystemType"
    value     = "enhanced"
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name      = "ConfigDocument"
    value     = file("cw_health_reporting.json")
  }
}

resource "aws_elastic_beanstalk_environment" "heimdall" {
  name          = "${var.application_name}-${var.environment_name}"
  application   = var.application_name
  template_name = aws_elastic_beanstalk_configuration_template.heimdall.name
}

################################
###### Instance Profiles #######
################################

resource "aws_iam_instance_profile" "heimdall" {
  name = "${var.application_name}-${var.environment_name}-ip"
  role = aws_iam_role.heimdall.name
}

resource "aws_iam_role" "heimdall" {
  name = "${var.application_name}-${var.environment_name}-role"
  assume_role_policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "sts:AssumeRole",
            "Principal": {
               "Service": "ec2.amazonaws.com"
            },
            "Effect": "Allow",
            "Sid": ""
        }
    ]
}
EOF
}

// A copy of the AWSElasticBeanstalkWebTier policy.
//For some reason policy attachment would often return an invalid arn, so we're creating this for it to reference
resource "aws_iam_policy" "AWSElasticBeanstalkWebTier" {
  name = "${var.application_name}-${var.environment_name}-AWSElasticBeanstalkWebTier"
  description = "Used for deploying code to beanstalk application"

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "BucketAccess",
            "Action": [
                "s3:Get*",
                "s3:List*",
                "s3:PutObject"
            ],
            "Effect": "Allow",
            "Resource": [
                "arn:aws:s3:::elasticbeanstalk-*",
                "arn:aws:s3:::elasticbeanstalk-*/*"
            ]
        },
        {
            "Sid": "XRayAccess",
            "Action": [
                "xray:PutTraceSegments",
                "xray:PutTelemetryRecords"
            ],
            "Effect": "Allow",
            "Resource": "*"
        },
        {
            "Sid": "CloudWatchLogsAccess",
            "Action": [
                "logs:PutLogEvents",
                "logs:CreateLogStream"
            ],
            "Effect": "Allow",
            "Resource": [
                "arn:aws:logs:*:*:log-group:/aws/elasticbeanstalk*"
            ]
        }
    ]
}
EOF
}

resource "aws_iam_policy_attachment" "beanstalk_web_tier" {
  name = "Beanstalk Web Tier"
  roles  = [aws_iam_role.heimdall.name]
  policy_arn = aws_iam_policy.AWSElasticBeanstalkWebTier.arn
}

// Metrics for AWS Memory Utilization Scripts
resource "aws_iam_policy" "metrics_monitoring" {
  name        = "MetricsMonitoring-${var.application_name}-${var.environment_name}"
  description = "Used to post and get metrics for EC2 instances to Cloudwatch for ${var.application_name} ${var.environment_name}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1498371131251",
      "Action": [
        "cloudwatch:GetMetricStatistics",
        "cloudwatch:ListMetrics",
        "cloudwatch:PutMetricData"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "Stmt1498371168662",
      "Action": [
        "ec2:DescribeTags"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_iam_policy_attachment" "metrics_monitoring" {
  name = "MetricsMonitoring-${var.application_name}-${var.environment_name}"
  roles  = [aws_iam_role.heimdall.name]
  policy_arn = aws_iam_policy.metrics_monitoring.arn
}

# Allow AWS SSM Access
resource "aws_iam_role_policy_attachment" "ssm" {
  role       = aws_iam_role.heimdall.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore"
}

# Allow Cloudwatch Logs Agent Publishing
resource "aws_iam_role_policy_attachment" "cw_logs_agent" {
  role       = aws_iam_role.heimdall.name
  policy_arn = "arn:aws:iam::aws:policy/CloudWatchAgentServerPolicy"
}
