# https://jira.twitch.com/browse/IPDEL-1612
# Creates a Proxy Jumphost to allow users via Teleport Bastion to ssh forward to Heimdall without being on VPN
resource "aws_instance" "proxy" {
  ami           = "ami-09c5e030f74651050"
  instance_type = "t3.micro"
  subnet_id     = data.aws_subnet.private_a.id
  vpc_security_group_ids = [
    data.aws_security_group.allow_ssh_from_twitch_subnets.id,
    aws_security_group.proxy.id,
    data.aws_security_group.default.id,
  ]
  iam_instance_profile = "Allow_SSM_Access"

  tags = {
    Name = "heimdall-jumphost"
  }

  # Configure sudoers and ssh to accept via teleport bastion / ldap
  user_data = <<USERDATA
#!/bin/bash

# Configure Sudoers
mkdir -p /etc/sudoers.d
cat <<EOT >> /etc/sudoers.d/twitch-ldap
%team-browser-grid ALL=(ALL) NOPASSWD:ALL
%security-vuln-mgmt-sudo ALL=(ALL) NOPASSWD:ALL
EOT
chmod 440 /etc/sudoers.d/twitch-ldap

# Configure SSH via LDAP & Teleport
mkdir -p /etc/sssd
cat <<EOT >> /etc/sssd/sssd.conf
[sssd]
config_file_version = 2
domains = justin.tv
services = nss,pam,ssh

[domain/justin.tv]
auth_provider = ldap
cache_credentials = true
debug_level = 0
enumerate = true
id_provider = ldap
ldap_group_member = uniqueMember
ldap_group_search_base = ou=Groups,dc=justin,dc=tv?onelevel?
ldap_id_use_start_tls = true
ldap_schema = rfc2307bis
ldap_search_base = dc=justin,dc=tv
ldap_tls_cacert = /etc/pki/tls/cert.pem
ldap_tls_reqcert = demand
# See https://wiki.twitch.com/pages/viewpage.action?pageId=81199247 - do not use ldap-vip.
ldap_uri = ldaps://ldap.twitch.a2z.com
ldap_user_search_base = ou=Users,dc=justin,dc=tv?onelevel?
min_id = 10000
refresh_expired_interval = 4050
EOT

chmod 400 /etc/sssd/sssd.conf

yum install -y sssd
systemctl enable sssd || chkconfig sssd on
service sssd start
authconfig --enablesssd --enablesssdauth --enablemkhomedir --updateall
sed -i 's+AuthorizedKeysCommand /opt/aws/bin/eic_run_authorized_keys %u %f+AuthorizedKeysCommand /usr/bin/sss_ssh_authorizedkeys+' /etc/ssh/sshd_config
sed -i 's+AuthorizedKeysCommandUser ec2-instance-connect+AuthorizedKeysCommandUser nobody+' /etc/ssh/sshd_config
service sshd restart

USERDATA

}

resource "aws_route53_record" "proxy" {
  zone_id = aws_route53_zone.heimdall.id
  name    = "proxy"
  type    = "A"
  ttl     = 300
  records = [aws_instance.proxy.private_ip]
}
