#!/usr/bin/groovy
package tv.justin.qe;

/**
 * Zip and upload an Artifact for Elastic Beanstalk
 * Use the elasticbeanstalk API to tag it appropriately and make it
 * available for deployment.
 *
 * This function assumes you are using the withCredentials wrapper
 * for your build:
 *
 * ```
 *   withCredentials([
       usernamePassword(credentialsId: 'my-aws-credential-id', usernameVariable: 'AWS_ACCESS_KEY_ID', passwordVariable: 'AWS_SECRET_ACCESS_KEY'),
     ])
 * ```
 *
 * Parameters:
 *  - applicationName - the identifier of the Beanstalk Application to install the artifact to
 *  - version - filename for the created artifact
 *  - s3bucket the bucket to upload to
 *  - archiveDirectory - relative path of the directory to be archived
 *
 * Example:
 *   createBeanstalkArtifact MyTestApplication \
 *                           abc123-master \
 *                           elasticbeanstalk-us-west-2-465569119046 \
 *                           .manta
 *
 *
 */
def createBeanstalkArtifactFromDirectory(applicationName, version, s3Bucket, archiveDirectory = "./") {
  sh """
    export AWS_DEFAULT_REGION=us-west-2
    cd ${archiveDirectory}
    zip -r ./${version}.zip ./
    description=`git log --pretty=format:"%an - %ad: %s" --abbrev-commit -1 | cut -c 1-200`
    aws s3 cp ${version}.zip s3://${s3Bucket}/apps/${applicationName}/${version}.zip
    aws elasticbeanstalk create-application-version --no-auto-create-application --application-name ${applicationName} --version-label ${version} --description "\$description" --source-bundle S3Bucket=${s3Bucket},S3Key=apps/${applicationName}/${version}.zip
  """
}

/**
 * Cleanup Builds, there's a 1000 artifact limit
 *
 * This function assumes you are using the credentialsBinding wrapper
 * for your build:
 *
 * ```
 *   wrappers {
 *     credentialsBinding {
 *       string 'AWS_ACCESS_KEY', 'credentials-name-for-your-aws-access-key'
 *       string 'AWS_SECRET_KEY', 'credentials-name-for-your-aws-secret-key'
 *     }
 *   }
 * ```
 *
 * Parameters:
 *  - applicationName - the identifier of the Beanstalk Application to install the artifact to
 *
 */
def cleanBeanstalkArtifacts(applicationName, versionsToKeep = 50) {
  shell """
set +x
export AWS_ACCESS_KEY_ID=\$AWS_ACCESS_KEY
export AWS_SECRET_ACCESS_KEY=\$AWS_SECRET_KEY
set -x
export AWS_DEFAULT_REGION=us-west-2
LIMIT=${versionsToKeep}
VERSIONS="\$(aws elasticbeanstalk describe-application-versions --application-name ${applicationName})"
VERSION_COUNT="\$(echo \${VERSIONS} | jq -r '.ApplicationVersions[] | .VersionLabel' | wc -l)"
if [ \${VERSION_COUNT} -gt \${LIMIT} ]; then
        DELETE_VERSION="\$(echo \${VERSIONS} | jq -r '.ApplicationVersions[] | .VersionLabel' | sed -n \${LIMIT}p)"
        if [ -n "\${DELETE_VERSION}" ]; then
                aws elasticbeanstalk delete-application-version \
                        --application-name ${applicationName} \
                        --version-label \${DELETE_VERSION}
        fi
fi
"""
}
