import com.onresolve.scriptrunner.runner.rest.common.CustomEndpointDelegate
import groovy.json.JsonBuilder
import groovy.json.JsonSlurper
import groovy.transform.BaseScript

import javax.ws.rs.core.MultivaluedMap
import javax.ws.rs.core.Response

// this line makes methods in your script recognisable as endpoints, and is required
@BaseScript CustomEndpointDelegate delegate

// Scope configuration into a class so that it's accessible consistently (eg Config.foo) everywhere in this script
class Config {
    final static URL SERVICE_CATALOG_ENDPOINT = new URL('https://servicecatalog-api.internal.justin.tv/api/v2/query')
    final static String LIST_SERVICE_NAMES_GRAPHQL = '{"operationName":"FetchAllServices", "variables":{}, "query":"query FetchAllServices {services{name, state}}"}'
    final static int RETRIES = 3
}

// Simpple backoff
def backOff(int attempt) {
    // sleep or exit
    if (attempt == Config.RETRIES) {
        return  // no need to sleep after final attempt
    }
    try {
        Thread.sleep(attempt*1000) // simple linear back-off mechanism
    } catch (InterruptedException ie) {
        ie.printStackTrace()
    }
}

// This method is exposed as an end-point. Maps automatically to route <script runner base url>/getTwitchServices
getTwitchServices(httpMethod: "GET") { MultivaluedMap queryParams, String body ->
    String output
    int statusCode
    for(int attempt = 1; attempt <= Config.RETRIES; attempt++) {
        try {
            URLConnection connection = Config.SERVICE_CATALOG_ENDPOINT.openConnection()
            connection.with {
                connectTimeout = 1000
                readTimeout = 2000
                doOutput = true
                requestMethod = 'POST'
                outputStream.withWriter { writer ->
                    writer << Config.LIST_SERVICE_NAMES_GRAPHQL
                }
                output = content.text
            }
            statusCode = connection.getResponseCode()
            if (statusCode >= 200 && statusCode < 400) {
                break  // break out of retry loop once there is a good response
            } else {
                backOff(attempt)  // otherwise, back off and retry
            }
        } catch (Exception e) {
            backOff(attempt) // backoff and retry on any exception
        }
    }

    if (statusCode >= 200 && statusCode < 400 && output != null && output.length() > 0) {
        def rawJson = new JsonSlurper().parseText(output)
        def finalJson = new JsonBuilder()
        List<String> query = (List<String>)queryParams.get("query")
        finalJson {
            items rawJson.data.services.findAll{
                it.state == "Active" && (query == null || query.size() == 0 || it.name.toLowerCase().contains(query.get(0).toLowerCase()))
            }.collect {
                [
                        label: it.name,
                        value: it.name,
                        html: it.name
                ]
            }
        }
        return Response.ok(finalJson.toString()).build()
    } else {
        Response.ok(new JsonBuilder([services: []]).toString()).build()  // empty response
    }
}
