import inspect
import os.path


class DelayedAssertion(object):

    def __init__(self):
        self.failed_expectations = []

    def __enter__(self):
        return self.expect

    def __exit__(self, *args):
        self.assert_expectations()

    def expect(self, expr, msg=None):
        'keeps track of failed expectations'
        if not expr:
            self.log_failure(msg)

    def assert_expectations(self):
        'raise an assert if there are any failed expectations'
        if self.failed_expectations:
            assert False, self.report_failures()

    def log_failure(self, msg=None):
        (filename, line, funcname, contextlist) =  inspect.stack()[2][1:5]
        filename = os.path.basename(filename)
        context = contextlist[0]
        self.failed_expectations.append('file "%s", line %s, in %s()%s\n%s' %
                                    (filename, line, funcname, (('\n%s' % msg) if msg else ''), context))

    def report_failures(self):
        if self.failed_expectations:
            (filename, line, funcname) =  inspect.stack()[2][1:4]
            report = [
                '\n\nassert_expectations() called from',
                '"%s" line %s, in %s()\n' % (os.path.basename(filename), line, funcname),
                'Failed Expectations:%s\n' % len(self.failed_expectations)]
            for i,failure in enumerate(self.failed_expectations, start=1):
                report.append('%d: %s' % (i, failure))
            self.failed_expectations = []
        return ('\n'.join(report))
