import base64
import os

import requests
from retrying import retry

from fwk.base.sauce.sauce_exception import SauceException

SAUCE_DEVICE_LIST = [('Android', 'Samsung Galaxy S8 HD GoogleAPI Emulator', '7.1', 'chrome'),
                     ('ios', 'iPhone X Simulator','11.1', 'safari')]
SAUCE_REST_END_POINT = 'https://saucelabs.com/rest/v1/'

# Get the sauce labs server url
def get_sauce_lab_url(user, key):
    """
    Get the sauce connection url.
    :param user:
    :param key:
    :return:
    """
    url = 'http://' + user + ':' + key + '@ondemand.saucelabs.com:80/wd/hub'
    return url


def get_sauce_credential():
    """
    Get the sauce credentials to connect to Sauce.
    :return:
    """
    return {'username': os.environ['sauce_username'], "password": os.environ['sauce_password']}


@retry(wait_exponential_multiplier=1000, wait_exponential_max=10000, stop_max_delay=60000)
def fetch_test_artifact(test_id, download_folder, artifact_type):
    """
    Fetch the screenshots and video from sauce.
    :param download_folder:
    :param artifact_type:
    :return:
    """
    get_job_assets(test_id, artifact_type, download_folder)


def get_job_assets(test_id, type, download_location):
    """
    Get job assets from sauce.
    :param download_location:
    :param test_id:
    :param type:
    :return:
    """
    credentials = get_sauce_credential()
    sauce_asset_end_point = SAUCE_REST_END_POINT + credentials['username'] + '/jobs/' + test_id + '/assets'
    if type == 'screenshots':
        screenshot_folder = download_location + '/' + 'screenshots'
        make_results_dir(screenshot_folder)
        filename =  screenshot_folder + '/screenshots.zip'
        sauce_url = sauce_asset_end_point + '/screenshots.zip'
    if type == 'video':
        filename = download_location + '/video.mp4'
        sauce_url = sauce_asset_end_point + '/video.mp4'
    headers = {"Authorization": "Basic %s" % encode_sauce_credentials(),
               'content_type':'application/json'}
    response = requests.get(sauce_url, headers=headers)
    if response.status_code not in [200, 201]:
        raise SauceException('{}: {}.\nSauce Status NOT OK'.format(
            response.status_code, response.reason))
    file = open(filename, 'wb')
    file.write(response.content)
    file.close


def encode_sauce_credentials():
    credentials = get_sauce_credential()
    username = credentials['username']
    password = credentials['password']
    auth_info = '{}:{}'.format(username, password)
    return base64.b64encode(auth_info.encode('utf-8')).decode('utf-8')

def make_results_dir(folder):
    if not os.path.exists(folder):
        os.makedirs(folder)
