package db

import (
	_ "github.com/go-sql-driver/mysql"
	"github.com/jmoiron/sqlx"

	log "code.justin.tv/qe/oml-service/logger"
	"code.justin.tv/qe/oml-service/util"

	"os"
	"net/http"
	"golang.org/x/net/context"
)

// NewDBUtil creates new DBUtil instance, which first creates connection to sql database. It also provides access to
// Convenience functions for querying the OML database.
type DBUtil struct {
	db *sqlx.DB
}

func NewDBUtil() (*DBUtil, error) {
	dbUtil := &DBUtil{}
	database, err := dbUtil.Connect()
	dbUtil.db = database

	return dbUtil, err
}

func (d *DBUtil) Connect() (*sqlx.DB, error) {
	dbConnectionString := os.Getenv("OML_DB_CONNECTION")
	var err error
	if len(dbConnectionString)==0 {
		secretsId := "qa-eng/oml/" + os.Getenv("ENVIRONMENT")  + "/oml_db_connection"
		dbConnectionString, err = util.GetSandstormHelperInstance().GetSecrets(secretsId)
		if err != nil {
			log.Error(err.Error(), "unable to query " + secretsId + " from sandstorm")
			return nil, err
		}
	}

	// DO NOT invoke "defer db.Close()" as the sql.DB object is designed to be long-lived.
	// Refer to http://go-database-sql.org/accessing.html
	database, err := sqlx.Open("mysql", dbConnectionString)

	if err != nil {
		log.Error(err.Error(), "unable to create db connection")
		return nil, err
	}

	if connectErr := database.Ping(); connectErr != nil {
		log.Error(connectErr.Error(), "unable to connect to db")
		return nil, err
	}

	return database, err
}

func (d *DBUtil) Handler(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		// Check if connection is still usable. If not usable, reconnect and update pointer.
		// Otherwise, continue using old pointer
		if connectErr := d.db.Ping(); connectErr != nil {
			log.Warn(connectErr.Error(), "Existing DB connection failed. Renewing connection.")
			database, err := d.Connect()
			if err != nil {
				log.Fatal("Could not renew connection")
			} else {
				d.db = database
			}
		}
		ctx := context.WithValue(context.Background(), "dbutil", d)
		wrappedRequest := r.WithContext(ctx)
		h.ServeHTTP(w, wrappedRequest)
	})
}
