package db

import (
	"testing"

	"github.com/DATA-DOG/go-sqlmock"
	"github.com/jmoiron/sqlx"

	"code.justin.tv/qe/oml-service/models"
)

func createMockDbUtil(t *testing.T) (*DBUtil, sqlmock.Sqlmock) {
	db, mock, err := sqlmock.New()
	if err != nil {
		t.Fatalf("unable to create database")
	}
	sqlxDB := sqlx.NewDb(db, "sqlmock")
	dbUtil := &DBUtil{
		db: sqlxDB,
	}
	return dbUtil, mock
}

func TestDBUtil_GetKpis(t *testing.T) {
	expected := models.Kpi{1, "KPI_KEY", "KPI1", 2}

	dbUtil, mock := createMockDbUtil(t)
	rows := sqlmock.NewRows([]string{"ID", "PRACTICE_ID", "KEY", "NAME"}).
		AddRow(expected.ID, expected.PracticeID, expected.Key, expected.Name)
	mock.ExpectQuery("SELECT ID, PRACTICE_ID, `KEY`, NAME FROM KPI").WillReturnRows(rows)

	kpis, err := dbUtil.GetKpis()
	if err != nil {
		t.Fatalf("unable to parse kpis")
	}

	if kpi := *kpis[0]; kpi != expected {
		t.Errorf("Expected: %v, given %v", expected, kpi)
		t.Fatalf("kpis returned do not match expected")
	}
}

func TestDBUtil_GetOrg(t *testing.T) {
	owner := "Ronald"
	expected := models.Org{1, "API", 1, "Visage", &owner}

	dbUtil, mock := createMockDbUtil(t)
	rows := sqlmock.NewRows([]string{"BU.ID", "BU.NAME", "T.ID", "T.NAME", "T.ENG_LEAD"}).
		AddRow(expected.OmlBuID, expected.OmlBuName, expected.OmlTeamID, expected.OmlTeamName, expected.Owner)
	mock.ExpectQuery(`SELECT .* FROM OML_TEAM T JOIN OML_BU BU ON \(T.OML_BU_ID=BU.ID\)`).
		WillReturnRows(rows)

	orgs, err := dbUtil.GetOrg(nil, nil)
	if err != nil {
		t.Fatalf("unable to parse org")
	}

	if org := *orgs[0]; org != expected {
		t.Errorf("Expected: %v, given %v", expected, org)
		t.Fatalf("org returned do not match expected")
	}

	mock.ExpectQuery(`BU.ID = ?`).WillReturnRows(sqlmock.NewRows([]string{}))
	var buId int64 = 42
	_, err = dbUtil.GetOrg(&buId, nil)
	if err != nil {
		t.Fatalf("Could not parse BUID")
	}

	mock.ExpectQuery(`T.ID = ?`).WillReturnRows(sqlmock.NewRows([]string{}))
	var TID int64 = 42
	_, err = dbUtil.GetOrg(nil, &TID)
	if err != nil {
		t.Fatalf("could not parse tid")
	}

	_, err = dbUtil.GetOrg(&buId, &TID)
	if err == nil {
		t.Fatalf("get_org failed to fail on passing in both buid and tid")
	}
}

func TestDBUtil_GetPractices(t *testing.T) {
	expected := models.Practice{1, "Eat your veggies"}

	dbUtil, mock := createMockDbUtil(t)
	rows := sqlmock.NewRows([]string{"ID", "NAME"}).
		AddRow(expected.ID, expected.Name)
	mock.ExpectQuery(`SELECT ID, NAME FROM PRACTICE`).
		WillReturnRows(rows)

	practices, err := dbUtil.GetPractices()
	if err != nil {
		t.Fatalf("unable to parse practices")
	}

	if practice := *practices[0]; expected != practice {
		t.Fatalf("Expected %v doesn't match retrieved %v", expected, practice)
	}
}

func TestDBUtil_GetMaxMetricsSummaryDate(t *testing.T) {
	expected := "28-NOV-2017"

	dbUtil, mock := createMockDbUtil(t)
	rows := sqlmock.NewRows([]string{"Date"}).
		AddRow(expected)
	mock.ExpectQuery(`MAX\(INGEST_BATCH\)`).
		WillReturnRows(rows)
	date, err := dbUtil.getLatestTimestampHelper(nil, nil)
	if err != nil {
		t.Fatalf(err.Error())
	}

	if expected != date {
		t.Fatalf("expected %v does not match given %v", expected, date)
	}
}

func TestDBUtil_GetRollups(t *testing.T) {
	expected := models.Rollup{1, "Department"}

	dbUtil, mock := createMockDbUtil(t)
	rows := sqlmock.NewRows([]string{"ID", "NAME"}).
		AddRow(expected.ID, expected.Name)
	mock.ExpectQuery(`SELECT .* FROM ROLLUP`).
		WillReturnRows(rows)
	rollups, err := dbUtil.GetRollups()
	if err != nil {
		t.Fatalf(err.Error())
	}

	if rollup := *rollups[0]; expected != rollup {
		t.Fatalf("expected %v does not match given %v", expected, rollup)
	}
}

func TestDBUtil_GetMetricsSummary(t *testing.T) {
	var gheId int64 = 1
	var buID int64 = 4
	var level int64 = 5
	var serviceId int64 = 6
	var teamId int64 = 7
	var rollupId int64 = 1
	expected := models.MetricsSummary{0.5, &gheId, "12-2-2017", 3, &buID,
		&level, &serviceId, &teamId, &rollupId, 5, 10}

	dbUtil, mock := createMockDbUtil(t)
	rows := sqlmock.NewRows([]string{"INGEST_BATCH", "ROLLUP_ID", "OML_BU_ID", "OML_TEAM_ID", "OML_SERVICE_ID",
		"GHE_REPO_ID", "KPI_ID", "VALUE1", "VALUE2", "COMPUTED_VALUE", "OML_LEVEL"}).
		AddRow(expected.IngestBatch, expected.RollupID, expected.OmlBuID, expected.OmlTeamID, expected.OmlServiceID,
			expected.GheRepoID, expected.KpiID, expected.Value1, expected.Value2, expected.ComputedValue, expected.OmlLevel)
	mock.ExpectQuery(`SELECT .* FROM METRICS_SUMMARY .* DATE\(INGEST_BATCH\) IN .* ROLLUP_ID = `).
		WillReturnRows(rows)
	metric_summaries, err := dbUtil.GetMetricsSummary(*expected.RollupID, expected.IngestBatch, expected.OmlBuID,
		expected.OmlTeamID)
	if err != nil {
		t.Fatalf(err.Error())
	}

	if m_s := *metric_summaries[0]; m_s != expected {
	    t.Errorf("Expected %v did not match given %v", expected, m_s)
	}

	rollupId = 2
	mock.ExpectQuery(`OML_BU_ID = ?`).WillReturnRows(sqlmock.NewRows([]string{}))
	_, err = dbUtil.GetMetricsSummary(*expected.RollupID, expected.IngestBatch, expected.OmlBuID,
		expected.OmlTeamID)
	if err != nil {
		t.Errorf("unable to retrieve metrics summary for bu level rollup")
		t.Fatalf(err.Error())
	}

	rollupId = 3
	mock.ExpectQuery(`OML_TEAM_ID = ?`).WillReturnRows(sqlmock.NewRows([]string{}))
	_, err = dbUtil.GetMetricsSummary(*expected.RollupID, expected.IngestBatch, expected.OmlBuID,
		expected.OmlTeamID)
	if err != nil {
		t.Errorf("unable to retrieve metrics summary for team level rollup")
		t.Fatalf(err.Error())
	}
}
