package db

import (
	sq "gopkg.in/Masterminds/squirrel.v1"

	log "code.justin.tv/qe/oml-service/logger"
	"code.justin.tv/qe/oml-service/models"
	"strings"
)

// GetMetricsSummary returns all data oml has for the given rollup(org, team, repo, etc.) level and ingestDate
func (dbUtil *DBUtil) GetMetricsSummary(rollupID int64, ingestDate string, omlBuID *int64, omlTeamID *int64) ([]*models.MetricsSummary, error) {
	query := sq.Select("INGEST_BATCH", "ROLLUP_ID", "OML_BU_ID", "OML_TEAM_ID", "OML_SERVICE_ID",
		"GHE_REPO_ID", "KPI_ID", "VALUE1", "VALUE2", "COMPUTED_VALUE", "OML_LEVEL").
		From("METRICS_SUMMARY").
		Where(sq.Eq{"DATE(INGEST_BATCH)": strings.Split(ingestDate, " ")}).
		Where(sq.Eq{"ROLLUP_ID": rollupID})

	switch rollupID {
	case 1:
		// Twitch level rollup
	case 2:
		// BU level rollup
		query = query.Where(sq.Eq{"OML_BU_ID": *omlBuID})
	case 3:
		// Team level rollup
		query = query.Where(sq.Eq{"OML_TEAM_ID": *omlTeamID})
	}

	queryString, args, err := query.ToSql()
	if err != nil {
		log.Error(err.Error(), "Could not Parse args into SQL in metrics summary")
		return nil, err
	}

	rows, err := dbUtil.db.Query(queryString, args...)
	if err != nil {
		log.Error(err.Error(), "Could not retrieve data on metrics summary from db")
		return nil, err
	}

	var metricSummaries []*models.MetricsSummary
	for rows.Next() {
		var metricSummary models.MetricsSummary

		err := rows.Scan(&metricSummary.IngestBatch, &metricSummary.RollupID, &metricSummary.OmlBuID,
			&metricSummary.OmlTeamID, &metricSummary.OmlServiceID, &metricSummary.GheRepoID, &metricSummary.KpiID,
			&metricSummary.Value1, &metricSummary.Value2, &metricSummary.ComputedValue, &metricSummary.OmlLevel)
		if err != nil {
			log.Error(err.Error(), "could not read metric Summary from DB")
			return nil, err
		}

		metricSummaries = append(metricSummaries, &metricSummary)
	}

	return metricSummaries, nil
}

// getLatestTimestampHelper performs the queries on the DB for timestamps.
// This private helper function returns error if it is unable to find the timestamp for the specified params.
func (dbUtil *DBUtil) getLatestTimestampHelper(kpiId *int64, ingestDate *string) (string, error) {
	selectBuilder := sq.Select("MAX(INGEST_BATCH)").From("METRICS_SUMMARY")
	if kpiId != nil {
		selectBuilder = selectBuilder.Where(sq.Eq{"KPI_ID": kpiId})
	}
	if ingestDate != nil {
		selectBuilder = selectBuilder.Where(sq.Eq{"DATE(INGEST_BATCH)": ingestDate})
	}

	query, args, err := selectBuilder.ToSql()

	if err != nil {
		log.Error(err.Error(), "unable to construct query")
		return "", err
	}
	row := dbUtil.db.QueryRow(query, args...)

	var timestamp string
	err = row.Scan(&timestamp)
	return timestamp, err
}

// GetLatestTimestamp returns the latest timestamp for the specified kpi on that specified date.
// If ingestDate is nil, this retrieves the absolute latest timestamp.
// If ingestDate is NOT nil, this retrieves the latest timestamp for that specific date.
func (dbUtil *DBUtil) GetLatestTimestamp(kpiId *int64, ingestDate *string) (string, error) {
	timestamp, err := dbUtil.getLatestTimestampHelper(kpiId, ingestDate)
	if ingestDate != nil && err != nil {
		// There was an error when date was specified, so try to get the latest available timestamp instead
		timestamp, err = dbUtil.getLatestTimestampHelper(kpiId, nil)
	}
	return timestamp, err
}

