package logger

import (
    "github.com/sirupsen/logrus"
    "github.com/heroku/rollrus"
    "log"
    "os"
    "sync"
)

const BUFFER_CAPACITY = 1000

func setupHooks(isAsync bool) {
    rollbarToken := os.Getenv("ROLLBAR_TOKEN")
    env := os.Getenv("ENVIRONMENT")
    if isAsync {
        logrus.AddHook(GetAsyncHook(rollbarToken, env))
    } else {
        logrus.AddHook(GetSyncHook(rollbarToken, env))
    }
}

var newRollrusHook = rollrus.NewHookForLevels
var triggerLevels = []logrus.Level{
logrus.ErrorLevel,
logrus.FatalLevel,
logrus.PanicLevel,
}

// var is here to allow changes for testing
var GetSyncHook = func (rollbarToken string, env string) *rollrus.Hook{
    if rollbarToken == "" || env == "" {
        log.Printf("Warning Rollbartoken and ENV not defined, not sending to Rollbar")
    }
    return newRollrusHook(rollbarToken, env, triggerLevels)
}

var getLogrusSyncHook = func(rollbarToken string, env string) logrus.Hook{
    return logrus.Hook(GetSyncHook(rollbarToken, env))
}

func GetAsyncHook(rollbarToken string, env string) *AsyncHook{
    syncHook := getLogrusSyncHook(rollbarToken, env)
    asyncHook := &AsyncHook{&syncHook, make(chan *logrus.Entry, BUFFER_CAPACITY), &sync.WaitGroup{}}
    asyncHook.StartWorker()
    return asyncHook
}

type AsyncHook struct {
   syncHook *logrus.Hook
   bodyChannel chan *logrus.Entry
   waitGroup *sync.WaitGroup
}

func (h *AsyncHook) StartWorker() {
   go func() {
       for entry := range h.bodyChannel {
           err := (*h.syncHook).Fire(entry)
           if err != nil {
               log.Printf("Error encountered while trying to log to Rollbar, %v", err)
               // TODO handle exponential backoff
           }
           h.waitGroup.Done()
       }
   }()
}

func (h *AsyncHook) Levels() []logrus.Level{
    return (*h.syncHook).Levels()
}

func(h *AsyncHook) Fire(entry *logrus.Entry) error {
    h.waitGroup.Add(1 )

    // capture stacktrace here and replace entry with pkg errors

    select {
    case h.bodyChannel <- entry:
    default:
        // TODO Allow configurable behavior around backing up
        log.Println("Rollbar Async Channel is backed up. Avoiding send to Rollbar by dropping log.")
        log.Printf("Dropped entry is %v", entry)
    }
    return nil
}

func(h *AsyncHook) Wait() {
    h.waitGroup.Wait()
}
