package logger

import (
    "github.com/pkg/errors"

    "runtime"

    "github.com/sirupsen/logrus"
    "strings"
    "fmt"
)

var filePrefix, fnPrefix string
const isAsync = false

// TODO Maybe. Allow users to supply a list of information through withfields and then get the basic set of loggers

func init() {
    logrus.SetLevel(logrus.DebugLevel)

    pc, file, _, ok := runtime.Caller(0)
    if !ok {
        logrus.Panic("failed to get runtime context")
        return
    }
    fn := runtime.FuncForPC(pc).Name()

    logrus.SetFormatter(&logrus.TextFormatter{
        FullTimestamp:    true,
        DisableTimestamp: false,
    })
    setupHooks(isAsync)

    filePrefix = strings.Split(file, "/log/log.go")[0] + "/"
    fnPrefix = strings.Split(fn, "/log.init")[0] + "/"
}

func Panic(v ...interface{}) {
    getBaseLogger(v...).Panic(v...)
}

func Fatal(v ...interface{}) {
    getBaseLogger(v...).Fatal(v...)
}

func Error(v ...interface{}) {
    getBaseLogger(v...).Error(v...)
}

// WithError will preserve the stack trace of an error if provided. Otherwise it will attach one to an error
func WithError(e error) {
    baseFields := getBaseFields()
    if _, ok := e.(stackTracer); ok {
        baseFields["err"] = e
    } else {
        baseFields["err"] = errors.WithStack(e)
    }
    logrus.WithFields(baseFields).Error(e.Error())
}

type stackTracer interface {
    StackTrace() errors.StackTrace
}

func Warn(v ...interface{}) {
    getBaseLogger(v...).Warn(v...)
}

func Info(v ...interface{}) {
    getBaseLogger(v...).Info(v...)
}

func Debug(v ...interface{}) {
    getBaseLogger(v...).Debug(v...)
}

func getBaseFields() logrus.Fields{
    pc, file, line, ok := runtime.Caller(2)
    if !ok {
        logrus.Panic("error getting runtime context for logrus output")
        return nil
    }
    fn := runtime.FuncForPC(pc).Name()
    fields := logrus.Fields{
        "file": strings.Replace(file, filePrefix, "", 1),
        "line": line,
        "func": strings.Replace(fn, fnPrefix, "", 1),
    }
    return fields
}

func getBaseLogger(v ...interface{}) *logrus.Entry{
    message := fmt.Sprint(v...)
    stacktraceError := errors.Errorf(message)
    baseFields := getBaseFields()
    baseFields["err"] = stacktraceError
    return logrus.WithFields(baseFields)
}

