
package restapi

import (
	"crypto/tls"
	"log"
	"net/http"
	"time"

	"github.com/dre1080/recover"
	errors "github.com/go-openapi/errors"
	runtime "github.com/go-openapi/runtime"
	"github.com/justinas/alice"
	graceful "github.com/tylerb/graceful"
	"github.com/rs/cors"

	"code.justin.tv/common/config"
	"code.justin.tv/common/gometrics"

	"code.justin.tv/qe/oml-service/restapi/operations"
	"code.justin.tv/qe/oml-service/restapi/operations/kpis"
	"code.justin.tv/qe/oml-service/restapi/operations/metrics_summary"
	"code.justin.tv/qe/oml-service/restapi/operations/org"
	"code.justin.tv/qe/oml-service/restapi/operations/practices"
	"code.justin.tv/qe/oml-service/restapi/operations/rollups"
	"code.justin.tv/qe/oml-service/restapi/operations/metrics"
	"code.justin.tv/qe/oml-service/restapi/operations/repos"
	"code.justin.tv/qe/oml-service/db"
)

// This file is safe to edit. Once it exists it will not be overwritten

//go:generate swagger generate server --target .. --name OML --spec ../swagger.yml

func configureFlags(api *operations.OMLAPI) {
	// api.CommandLineOptionsGroups = []swag.CommandLineOptionsGroup{ ... }
}

func configureAPI(api *operations.OMLAPI) http.Handler {
	// configure the api here
	api.ServeError = errors.ServeError

	// Set your custom logger if needed. Default one is log.Printf
	// Expected interface func(string, ...interface{})
	//
	// Example:
	// api.Logger = log.Printf

	api.JSONConsumer = runtime.JSONConsumer()

	api.JSONProducer = runtime.JSONProducer()

	api.KpisGetKpisHandler = kpis.GetKpisHandlerFunc(kpis.Kpis)
	api.OrgGetOrgHandler = org.GetOrgHandlerFunc(org.Org)
	api.MetricsSummaryGetMetricsSummaryHandler = metrics_summary.GetMetricsSummaryHandlerFunc(metrics_summary.MetricsSummary)
	api.PracticesGetPracticesHandler = practices.GetPracticesHandlerFunc(practices.Practices)
	api.RollupsGetRollupsHandler = rollups.GetRollupsHandlerFunc(rollups.Rollups)
	api.MetricsGetMetricsHandler = metrics.GetMetricsHandlerFunc(metrics.Metrics)
	api.ReposGetReposHandler = repos.GetReposHandlerFunc(repos.Repos)
	api.GetHandler = operations.GetHandlerFunc(operations.DummyGet)

	api.ServerShutdown = func() {}

	return setupGlobalMiddleware(api.Serve(setupMiddlewares))
}

// The TLS configuration before HTTPS server starts.
func configureTLS(tlsConfig *tls.Config) {
	// Make all necessary changes to the TLS configuration here.
}

// As soon as server is initialized but not run yet, this function will be called.
// If you need to modify a config, store server instance to stop it individually later, this is the place.
// This function can be called multiple times, depending on the number of serving schemes.
// scheme value will be set accordingly: "http", "https" or "unix"
func configureServer(s *graceful.Server, scheme, addr string) {
}

// The middleware configuration is for the handler executors. These do not apply to the swagger.json document.
// The middleware executes after routing but before authentication, binding and validation
func setupMiddlewares(handler http.Handler) http.Handler {
	return handler
}


// The middleware configuration happens before anything, this middleware also applies to serving the swagger.json document.
// So this is a good place to plug in a panic handling middleware, logging and metrics
func setupGlobalMiddleware(handler http.Handler) http.Handler {
	gometrics.Monitor(config.Statsd(), time.Second*5)

	handlePanicOutput := recover.New(&recover.Options{
		Log: log.Print,
	})
	handleCORS := cors.Default().Handler
	dbUtil, err := db.NewDBUtil()
	if err != nil {
		log.Fatal("could not connect to DB during app startup")
	}
	dbContextHandler := dbUtil.Handler
	chain := alice.New(handlePanicOutput, LogHandler, PanicHandler, handleCORS, dbContextHandler).Then(handler)
	return chain
}
