package restapi

import (
	"net/http"
	"time"

	"github.com/sirupsen/logrus"
	"code.justin.tv/qe/oml-service/logger"
	"os"
)

const logFormat = "%s - [%s] \"%s %s %s\" %d %d - %dms"

type loggedResponseWriter struct {
	http.ResponseWriter
	status int
	data   []byte
}

func PanicHandler(next http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		rollbarToken := os.Getenv("ROLLBAR_TOKEN")
		env := os.Getenv("ENVIRONMENT")
		syncLogger := logger.GetSyncHook(rollbarToken, env)
		// Panics cannot be caught by functions more than one callstack away
		defer syncLogger.ReportPanic()
		next.ServeHTTP(w, r)
	})
}

func LogHandler(next http.Handler) http.Handler {

	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		start := time.Now().UTC()

		next.ServeHTTP(w, r)

		logged := &loggedResponseWriter{
			ResponseWriter: w,
		}
		logrus.Infof(
			logFormat,
			r.RemoteAddr,
			start.Format("02/Jan/2006 03:04:05"),
			r.Method,
			r.RequestURI,
			r.Proto,
			logged.status,
			len(logged.data),
			time.Since(start)/time.Millisecond,
		)

	})
}

func (l *loggedResponseWriter) Write(data []byte) (int, error) {
	l.data = data
	return l.ResponseWriter.Write(data)
}

func (l *loggedResponseWriter) WriteHeader(status int) {
	l.status = status
	l.ResponseWriter.WriteHeader(status)
}
