package util

import (
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"code.justin.tv/systems/sandstorm/manager"
	"time"
	"github.com/aws/aws-sdk-go/service/sts"
	"sync"
	log "code.justin.tv/qe/oml-service/logger"
	"os"
)

type SandstormHelper struct {
	roleArn string
	sandstormManager *manager.Manager
}

var instance *SandstormHelper
var once sync.Once

// GetSandstormHelperInstance returns the singleton instance of SandstormHelper.
// This uses reference implementation from http://marcio.io/2015/07/singleton-pattern-in-go/
func GetSandstormHelperInstance() *SandstormHelper {
	once.Do(func() {
		env := os.Getenv("ENVIRONMENT") // devtools' tooling on terraform + beanstalk provides this
		//Reference: roleArn should be the IAM role that has the sts:assumeRole policy
		//The role should be defined in terraform/twitch-cape-qe-aws/sandstorm_<env>.tf
		roleArn := "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/qa-" + env + "-oml-service"

		instance = &SandstormHelper{
			roleArn: roleArn,
		}
		instance.Init()
	})
	return instance
}

func (m *SandstormHelper) Init() {
	// Reference implementation from: https://git-aws.internal.justin.tv/systems/sandstorm/tree/master/manager
	// Config for us-west-2 for STS
	awsConfig := &aws.Config{Region: aws.String("us-west-2")} //TODO: use config package to get AWS region
	stsClient := sts.New(session.New(awsConfig))

	arp := &stscreds.AssumeRoleProvider{  //TODO: use config package to get these
		Duration: 900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN: m.roleArn,
		Client: stsClient,
	}

	// Create credentials and config using our Assumed Role that we will use to access the main account with
	omlCredentials := credentials.NewCredentials(arp)
	awsConfig.WithCredentials(omlCredentials)

	// Finally, create a secret manager that uses the cross-account config.
	m.sandstormManager = manager.New(manager.Config{
		AWSConfig:   awsConfig,
		// Reference: service name as defined tcs.yaml. Context: tcs tool was used to create the service.
		// See https://git-aws.internal.justin.tv/dta/tcs/tree/master/code_labs/getting_started)
		ServiceName: "oml-service",
	})

	// Enable caching within sandstorm manager, saving 2 network roundtrips by GetSecrets()
	// Reference: https://git-aws.internal.justin.tv/systems/sandstorm/tree/master/manager
	m.sandstormManager.ListenForUpdates()
}

// GetSecrets queries Sandstorm for specified secret and returns a string representation of the secret
func (m *SandstormHelper) GetSecrets(secretKeyName string) (string, error) {
	secret, err := m.sandstormManager.Get(secretKeyName)
	if err != nil {
		log.Error(err.Error())
		return "", err
	}
	return string(secret.Plaintext), nil
}