"""initial tables

Revision ID: 1485fa1f7f70
Revises: 
Create Date: 2017-09-19 11:28:07.756928

"""
import csv
import traceback

import sys

import requests
from alembic import op
import sqlalchemy as sa

# revision identifiers, used by Alembic.
from sqlalchemy import ForeignKey

from datasource import OMLCapacity

revision = '1485fa1f7f70'
down_revision = None
branch_labels = None
depends_on = None


def upgrade():
    """
    This is the initial version of OML schema.

    Some guiding thoughts/assumptions:
    1) Org structure in external systems (jira, status dashboard, github, etc) *will* change unannounced. Such
       changes must not impact the consistency and correctness of Org structure in OML's view.
    2) Mitigate other systems' Org structure changes with by mapping them to OML. All OML reports should be based on
       Org structure from OML's point of view.
    3) Changes can be reflected gracefully on OML's side by adding new entries and incorporating them in our queries
    4) Assume there will be changes to Metrics based on company needs
    5) Assume SLAs will be defined at various levels (Twitch, BU, Team, Service)
    6) Real-time updates are *not* required. Data ingestion will be on reasonable schedule (per day, per week, etc),
    7) This is not a high-volume data store. The largest growth will be metrics_entry table, so be aware of it when
        patching this table.
    """

    oml_bu = op.create_table(
        'OML_BU',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False, unique=True)
    )

    ghe_org = op.create_table(
        'GHE_ORG',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False, unique=True),
        sa.Column('OML_BU_ID', sa.Integer, ForeignKey('OML_BU.ID'), nullable=False),
        sa.Column('EXT_ID', sa.Integer, nullable=True)
    )

    oml_team = op.create_table(
        'OML_TEAM',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False),
        sa.Column('ENG_LEAD', sa.String(50), nullable=True),
        sa.Column('OML_BU_ID', sa.Integer, ForeignKey('OML_BU.ID'), nullable=False)
    )

    ghe_repo = op.create_table(
        'GHE_REPO',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False),
        sa.Column('OML_TEAM_ID', sa.Integer, ForeignKey('OML_TEAM.ID'), nullable=False),
        sa.Column('EXT_ID', sa.Integer, nullable=True),
    )

    jira_category = op.create_table(
        'JIRA_CATEGORY',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False, unique=True),
        sa.Column('OML_BU_ID', sa.Integer, ForeignKey('OML_BU.ID'), nullable=False)
    )

    jira_project = op.create_table(
        'JIRA_PROJECT',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('KEY', sa.String(16), nullable=False, unique=True),
        sa.Column('NAME', sa.String(50), nullable=False, unique=True),
        sa.Column('OML_TEAM_ID', sa.Integer, ForeignKey('OML_TEAM.ID'), nullable=False)
    )

    oml_service = op.create_table(
        'OML_SERVICE',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False),
        sa.Column('OML_TEAM_ID', sa.Integer, ForeignKey('OML_TEAM.ID'), nullable=False)
    )

    rollup_level = op.create_table(
        'ROLLUP',
         sa.Column('ID', sa.Integer, primary_key=True, nullable=False),
         sa.Column('NAME', sa.String(50), nullable=False, unique=True)
    )

    practice = op.create_table(
        'PRACTICE',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=False),
        sa.Column('NAME', sa.String(100), nullable=False, unique=True)
    )

    kpi = op.create_table(
        'KPI',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=False),
        sa.Column('PRACTICE_ID', sa.Integer, ForeignKey('PRACTICE.ID'), nullable=False),
        sa.Column('KEY', sa.String(12), nullable=False, unique=True),
        sa.Column('NAME', sa.String(100), nullable=False, unique=True)
    )

    goal = op.create_table(
        'GOAL',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('KPI_ID', sa.Integer, ForeignKey('KPI.ID'), nullable=False),
        sa.Column('ROLLUP_ID', sa.Integer, ForeignKey('ROLLUP.ID'), nullable=False),
        sa.Column('OML_LEVEL', sa.Integer, nullable=False),
        sa.Column('MIN_VALUE', sa.Float, nullable=True),
        sa.Column('MAX_VALUE', sa.Float, nullable=True)
    )

    time_series_def = op.create_table(
        'TIME_SERIES_DEF',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('KEY', sa.String(16), unique=True, nullable=False),
        sa.Column('NAME', sa.String(100), nullable=False)
    )

    time_series_raw = op.create_table(
        'TIME_SERIES_RAW',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('INGEST_BATCH', sa.DateTime, nullable=False),
        sa.Column('VALUE', sa.Float, nullable=False),
        sa.Column('OML_BU_ID', sa.Integer, ForeignKey('OML_BU.ID'), nullable=False),
        sa.Column('OML_TEAM_ID', sa.Integer, ForeignKey('OML_TEAM.ID'), nullable=True),
        sa.Column('OML_SERVICE_ID', sa.Integer, ForeignKey('OML_SERVICE.ID'), nullable=True),
        sa.Column('GHE_REPO_ID', sa.Integer, ForeignKey('GHE_REPO.ID'), nullable=True),
        sa.Column('TIME_SERIES_DEF_ID', sa.Integer, ForeignKey('TIME_SERIES_DEF.ID'), nullable=False)
    )

    metrics_summary = op.create_table(
        'METRICS_SUMMARY',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('KPI_ID', sa.Integer, ForeignKey('KPI.ID'), nullable=False),
        sa.Column('INGEST_BATCH', sa.DateTime, nullable=False),
        sa.Column('ROLLUP_ID', sa.Integer, ForeignKey('ROLLUP.ID'), nullable=False),
        sa.Column('OML_BU_ID', sa.Integer, ForeignKey('OML_BU.ID'), nullable=True),
        sa.Column('OML_TEAM_ID', sa.Integer, ForeignKey('OML_TEAM.ID'), nullable=True),
        sa.Column('OML_SERVICE_ID', sa.Integer, ForeignKey('OML_SERVICE.ID'), nullable=True),
        sa.Column('GHE_REPO_ID', sa.Integer, ForeignKey('GHE_REPO.ID'), nullable=True),
        sa.Column('VALUE1', sa.Float, nullable=False),
        sa.Column('VALUE2', sa.Float, nullable=False),
        sa.Column('COMPUTED_VALUE', sa.Float, nullable=False),
        sa.Column('OML_LEVEL', sa.Integer, nullable=True)
    )

    status_bu = op.create_table(
        'STATUS_BU',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False, unique=True),
        sa.Column('OML_BU_ID', sa.Integer, ForeignKey('OML_BU.ID'), nullable=False),
        sa.Column('EXT_ID', sa.Integer, nullable=True),
        sa.Column('EXT_KEY', sa.String(20), nullable=True),
    )

    status_service = op.create_table(
        'STATUS_SERVICE',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False, unique=True),
        sa.Column('OML_SERVICE_ID', sa.Integer, ForeignKey('OML_SERVICE.ID'), nullable=True),
        sa.Column('OML_TEAM_ID', sa.Integer, ForeignKey('OML_TEAM.ID'), nullable=True),
        sa.Column('EXT_ID', sa.Integer, nullable=True)
    )

    status_team = op.create_table(
        'STATUS_TEAM',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False, unique=True),
        sa.Column('OML_TEAM_ID', sa.Integer, ForeignKey('OML_TEAM.ID'), nullable=True),
        sa.Column('EXT_ID', sa.Integer, nullable=True)
    )

    incident_org = op.create_table(
        'INCIDENT_ORG',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False, unique=True),
        sa.Column('OML_BU_ID', sa.Integer, ForeignKey('OML_BU.ID'), nullable=True)
    )

    incident_org = op.create_table(
        'INCIDENT_COMP',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False, unique=True),
        sa.Column('OML_TEAM_ID', sa.Integer, ForeignKey('OML_TEAM.ID'), nullable=True)
    )

    capacity_bu = op.create_table(
        'CAPACITY_BU',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False, unique=True),
        sa.Column('OML_BU_ID', sa.Integer, ForeignKey('OML_BU.ID'), nullable=False),
        sa.Column('EXT_KEY', sa.String(20), nullable=True)
    )

    capacity_group = op.create_table(
        'CAPACITY_GROUP',
        sa.Column('ID', sa.Integer, primary_key=True, nullable=False, autoincrement=True),
        sa.Column('NAME', sa.String(50), nullable=False, unique=True),
        sa.Column('OML_TEAM_ID', sa.Integer, ForeignKey('OML_TEAM.ID'), nullable=True),
        sa.Column('EXT_KEY', sa.String(20), nullable=True),
        sa.Column('CAPACITY_BU_ID', sa.Integer, ForeignKey('CAPACITY_BU.ID'), nullable=False)
    )

    try:
        conn = op.get_bind()

        # add initial time series definitions
        # TODO: Web UI should allow manager/admin to create new time series definitions
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('RI-ALL', 'All RIs are closed within SLA - created')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('RI-CLOSED', 'All RIs are closed within SLA - resolved')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('S123ALL', 'All Sev-1 through Sev-3 bugs should be resolved in a timely manner - created')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('S123CLOSED', 'All Sev-1 through Sev-3 bugs should be resolved in a timely manner - resolved')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('INC-ALRT-1K', 'All incidents with Alerts')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('INC-ALL-1K', 'All incidents')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('AVAIL', 'Availability')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('AVAIL-SLA', 'Availability - SLA')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('CC-LOC-HIT', 'Code Coverage - LoC - Hit')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('CC-LOC-ALL', 'Code Coverage - LoC - Total')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('CFG-MANAGED', 'Config Management - Managed Hosts')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('CFG-TOTAL', 'Config Management - Total Hosts')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('UNITTEST', 'Unit Tested')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('CANARY', 'Canary Environment')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('AEC', 'Services Have Automated Environment Creation')")
        conn.execute("INSERT INTO TIME_SERIES_DEF(`KEY`, NAME) VALUES('INTEGTEST', 'Deployments are Integration Tested')")

        # Initial set of seed rollup levels. This is not expected to change much.
        conn.execute("INSERT INTO ROLLUP(`ID`, `NAME`) VALUES(1, 'Twitch')")
        conn.execute("INSERT INTO ROLLUP(`ID`, `NAME`) VALUES(2, 'BU')")
        conn.execute("INSERT INTO ROLLUP(`ID`, `NAME`) VALUES(3, 'Team')")
        conn.execute("INSERT INTO ROLLUP(`ID`, `NAME`) VALUES(4, 'Service')")
        conn.execute("INSERT INTO ROLLUP(`ID`, `NAME`) VALUES(5, 'Repo')")

        # Initial set of seed OML practices
        conn.execute("INSERT INTO PRACTICE(`ID`, `NAME`) VALUES(0, 'All services operate w/in an Availability SLA')")
        conn.execute("INSERT INTO PRACTICE(`ID`, `NAME`) VALUES(1, 'All RIs are closed within SLA')")
        conn.execute("INSERT INTO PRACTICE(`ID`, `NAME`) VALUES(2, 'All Bugs are closed within SLA')")
        conn.execute("INSERT INTO PRACTICE(`ID`, `NAME`) VALUES(3, 'All deployments use canaries')")
        conn.execute("INSERT INTO PRACTICE(`ID`, `NAME`) VALUES(4, 'All services track code coverage')")
        conn.execute("INSERT INTO PRACTICE(`ID`, `NAME`) VALUES(5, 'All hosts use automated configuration mgmt')")
        conn.execute("INSERT INTO PRACTICE(`ID`, `NAME`) VALUES(6, 'All services have Automated Env. Creation')")
        conn.execute("INSERT INTO PRACTICE(`ID`, `NAME`) VALUES(7, 'All deployments are Integration Tested')")
        conn.execute("INSERT INTO PRACTICE(`ID`, `NAME`) VALUES(8, 'All builds have automated tests (CI)')")
        conn.execute("INSERT INTO PRACTICE(`ID`, `NAME`) VALUES(9, 'All incidents have alerts')")

        # Initial KPIs
        conn.execute("INSERT INTO KPI(`ID`, `PRACTICE_ID`, `KEY`, `NAME`) VALUES(0, 0, 'AVAIL-SLA', '%% of Svcs w/in SLA')")
        conn.execute("INSERT INTO KPI(`ID`, `PRACTICE_ID`, `KEY`, `NAME`) VALUES(1, 1, 'RI-SLA', '%% RIs w/in SLA')")
        conn.execute("INSERT INTO KPI(`ID`, `PRACTICE_ID`, `KEY`, `NAME`) VALUES(2, 2, 'BUGS-SLA', '%% Bugs w/in SLA')")
        conn.execute("INSERT INTO KPI(`ID`, `PRACTICE_ID`, `KEY`, `NAME`) VALUES(3, 3, 'CANARIES', '%% Deploys w/ Canary')")
        conn.execute("INSERT INTO KPI(`ID`, `PRACTICE_ID`, `KEY`, `NAME`) VALUES(4, 4, 'CODECOV', '%% of Svcs w/ Coverage')")
        conn.execute("INSERT INTO KPI(`ID`, `PRACTICE_ID`, `KEY`, `NAME`) VALUES(5, 5, 'CFGMGMT', '%% Hosts w/ CM')")
        conn.execute("INSERT INTO KPI(`ID`, `PRACTICE_ID`, `KEY`, `NAME`) VALUES(6, 6, 'AEC', '%% Svcs w/ AEC')")
        conn.execute("INSERT INTO KPI(`ID`, `PRACTICE_ID`, `KEY`, `NAME`) VALUES(7, 7, 'INTEGTEST', '%% Deploys w/ Int Test')")
        conn.execute("INSERT INTO KPI(`ID`, `PRACTICE_ID`, `KEY`, `NAME`) VALUES(8, 8, 'UNITTEST', '%% Builds w/ Tests')")
        conn.execute("INSERT INTO KPI(`ID`, `PRACTICE_ID`, `KEY`, `NAME`) VALUES(9, 9, 'INC-ALERTS', '%% Incidents w/ Alerts')")
        
        # Initial set of goals - In the future this should be managed by a UI
        # Reference: https://docs.google.com/spreadsheets/d/1V30Esz5aT8W9-3SFwkcOEJtbYJBP91Ak0CgNJkKAvR8/edit?ts=59ced150#gid=0
        # Allow the goals to be very fine-grained (BU level / team level / etc) to allow BUs, Teams, etc flexibility
        for rollup_id in range(1, 6):
            # AVAIL-SLA
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(0, %s, 0, 0.0, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(0, %s, 1, 0.1, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(0, %s, 2, 0.25, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(0, %s, 3, 0.5, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(0, %s, 4, 0.8, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(0, %s, 5, 0.9, NULL)" % rollup_id)

            # RI-SLA
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(1, %s, 0, 0.0, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(1, %s, 1, 0.1, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(1, %s, 2, 0.25, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(1, %s, 3, 0.5, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(1, %s, 4, 0.8, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(1, %s, 5, 0.9, NULL)" % rollup_id)

            # BUGS-SLA
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(2, %s, 0, 0.0, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(2, %s, 1, 0.1, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(2, %s, 2, 0.25, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(2, %s, 3, 0.5, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(2, %s, 4, 0.8, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(2, %s, 5, 0.9, NULL)" % rollup_id)

            # CANARIES
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(3, %s, 0, 0.0, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(3, %s, 1, 0.1, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(3, %s, 2, 0.25, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(3, %s, 3, 0.5, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(3, %s, 4, 0.8, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(3, %s, 5, 0.9, NULL)" % rollup_id)

            # CODECOV
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(4, %s, 0, 0.0, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(4, %s, 1, 0.1, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(4, %s, 2, 0.25, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(4, %s, 3, 0.5, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(4, %s, 4, 0.8, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(4, %s, 5, 0.9, NULL)" % rollup_id)

            # CFGMGMT
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(5, %s, 0, 0.0, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(5, %s, 1, 0.1, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(5, %s, 2, 0.25, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(5, %s, 3, 0.5, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(5, %s, 4, 0.8, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(5, %s, 5, 0.9, NULL)" % rollup_id)

            # AEC
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(6, %s, 0, 0.0, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(6, %s, 1, 0.1, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(6, %s, 2, 0.25, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(6, %s, 3, 0.5, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(6, %s, 4, 0.8, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(6, %s, 5, 0.9, NULL)" % rollup_id)

            # INTEGTEST
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(7, %s, 0, 0.0, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(7, %s, 1, 0.1, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(7, %s, 2, 0.25, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(7, %s, 3, 0.5, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(7, %s, 4, 0.8, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(7, %s, 5, 0.9, NULL)" % rollup_id)

            # UNITTEST
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(8, %s, 0, 0.0, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(8, %s, 1, 0.1, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(8, %s, 2, 0.25, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(8, %s, 3, 0.5, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(8, %s, 4, 0.8, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(8, %s, 5, 0.9, NULL)" % rollup_id)

            # INC-ALERTS
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(9, %s, 0, 0.0, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(9, %s, 1, 0.1, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(9, %s, 2, 0.25, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(9, %s, 3, 0.5, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(9, %s, 4, 0.8, NULL)" % rollup_id)
            conn.execute("INSERT INTO GOAL(`KPI_ID`, `ROLLUP_ID`, `OML_LEVEL`, `MIN_VALUE`, `MAX_VALUE`) "
                         "VALUES(9, %s, 5, 0.9, NULL)" % rollup_id)

    except Exception as e:
        print("ERROR: DB Migration failed due to the error below. Perform a downgrade immediately and fix the issue!")
        traceback.print_exc(file=sys.stdout)


def downgrade():
    op.drop_table("CAPACITY_GROUP")
    op.drop_table("CAPACITY_BU")
    op.drop_table('INCIDENT_COMP')
    op.drop_table('INCIDENT_ORG')
    op.drop_table('STATUS_TEAM')
    op.drop_table('STATUS_SERVICE')
    op.drop_table('STATUS_BU')
    op.drop_table('METRICS_SUMMARY')
    op.drop_table('TIME_SERIES_RAW')
    op.drop_table('TIME_SERIES_DEF')
    op.drop_table('GOAL')
    op.drop_table('KPI')
    op.drop_table('PRACTICE')
    op.drop_table('OML_SERVICE')
    op.drop_table('JIRA_PROJECT')
    op.drop_table('JIRA_CATEGORY')
    op.drop_table('GHE_REPO')
    op.drop_table('GHE_ORG')
    op.drop_table('OML_TEAM')
    op.drop_table('OML_BU')
    op.drop_table('ROLLUP')
