#!/usr/bin/env python3.6

import argparse

import datetime
import pygsheets

import time

import sys

from handlers import DBHelper
from orghelper import OrgHelper


MAX_ROWS = 1000
MAX_COLS = 53  # 1 col per week in a year, plus 1 label col
BACKOFF_RETRY_WAIT = 20  # multiplier (in seconds) for backoff retry algorithm


def get_worksheet(worksheet_name):
    client = pygsheets.authorize(service_file="client_secret.json")

    sh = None
    for attempt in range(0, 3):  # retry 3 times with back-off algorithm
        try:
            sh = client.open("OML-Report")  # TODO: set this to a final place when decided
            break
        except Exception as e:
            time.sleep(BACKOFF_RETRY_WAIT * attempt)  # simple backoff algorithm

    output_sheet = None
    worksheets = None
    for attempt in range(0, 3):  # retry 3 times with back-off algorithm
        try:
            worksheets = sh.worksheets()
            break
        except Exception as e:
            time.sleep(BACKOFF_RETRY_WAIT * attempt)  # simple backoff algorithm

    for s in worksheets:
        if s.title == worksheet_name:
            output_sheet = s
            break

    if output_sheet is None:
        for attempt in range(0, 3):  # retry 3 times with back-off algorithm
            try:
                output_sheet = sh.add_worksheet(worksheet_name, MAX_ROWS, MAX_COLS)
                break
            except Exception as e:
                time.sleep(BACKOFF_RETRY_WAIT * attempt)  # simple backoff algorithm

    return output_sheet


FIRST_DATA_COL = 2
WEEK_OFFSET = 42  # 42st week will be treated as the "1st" week in the spreadsheet, and will be added at FIRST_DATA_COL


def update_column_values(header_col, header_value, data_col, data_value=''):
    header_col.append(header_value)
    data_col.append(data_value)


KPIS_ENABLED_FOR_SUMMARY = [0, 1, 2, 3, 4, 5, 8, 9]
#KPIS_ENABLED_FOR_SUMMARY = [0, 1, 2, 4, 9]

MAX_LEVEL = 999


def output_summary(args):
    # Add new column with metrics per week.
    # This means running multiple times in the same week will update the same column (a good thing?)
    sample_date = getattr(args, 'date', None)
    if sample_date is None:
        sample_date = time.strftime('%m/%d/%Y', time.localtime())
        date_clause = ""
        week_of_year = datetime.datetime.utcnow().isocalendar()[1]
    else:
        date_clause = " AND INGEST_BATCH <= '%s' " % sample_date
        week_of_year = datetime.datetime.strptime(("%s 23:59:59" % sample_date), '%Y/%m/%d %H:%M:%S').isocalendar()[1]

    header_col = []
    data_col = []

    if args.rollup == "twitch":
        print("Twitch")
        update_column_values(header_col, "Twitch", data_col, sample_date)
        practices = DBHelper.run_query("SELECT ID, NAME FROM PRACTICE ORDER BY ID")
        company_level = MAX_LEVEL  # Start with Max level, and iteratively find the lowest OML level

        for practice in practices:
            practice_id = practice[0]
            kpi = DBHelper.run_query_flat_results("SELECT ID,NAME FROM KPI WHERE PRACTICE_ID=%s" % practice_id)
            kpi_id = kpi[0]
            kpi_name = kpi[1]
            practice_rollup_twitch = DBHelper.run_query_flat_results("SELECT VALUE1,VALUE2,COMPUTED_VALUE,OML_LEVEL "
                                                                     "FROM METRICS_SUMMARY "
                                                                     "WHERE ROLLUP_ID=1 AND KPI_ID=%s %s "
                                                                     "ORDER BY INGEST_BATCH DESC LIMIT 1"
                                                                     % (kpi_id, date_clause))
            if len(practice_rollup_twitch) > 0:
                if kpi_id in KPIS_ENABLED_FOR_SUMMARY:
                    company_level = min(company_level, practice_rollup_twitch[3])
                    print("%s: %d/%d (%0.2f%%) [level %s]" % (kpi_name, practice_rollup_twitch[0],
                                                              practice_rollup_twitch[1], 100*practice_rollup_twitch[2],
                                                              practice_rollup_twitch[3]))
                    update_column_values(header_col, kpi_name, data_col, "%d/%d (%0.2f%%)"
                               % (practice_rollup_twitch[0], practice_rollup_twitch[1], 100*practice_rollup_twitch[2]))
                else:
                    print("%s: N/A" % kpi_name)
                    update_column_values(header_col, kpi_name, data_col, "N/A")
            else:
                print("%s: N/A" % kpi_name)
                update_column_values(header_col, kpi_name, data_col, "N/A")

        print("OML Level: %s" % (company_level if 0 <= company_level <= 5 else 0 if company_level> 5 else "N/A"))
        update_column_values(header_col, "OML Level", data_col, company_level if 0 <= company_level <= 5 else 0 if company_level> 5 else "N/A")

    else:
        if not hasattr(args, "bu"):
            print("--bu is required for rollup=bu")
            sys.exit(1)

        bu_id = DBHelper.run_query_flat_results("SELECT ID FROM OML_BU WHERE NAME='%s'" % args.bu)[0]
        print(args.bu)
        update_column_values(header_col, args.bu, data_col, time.strftime('%m/%d/%Y', time.localtime()))
        practices = DBHelper.run_query("SELECT ID, NAME FROM PRACTICE ORDER BY ID")
        bu_oml_level = MAX_LEVEL  # Start with Max level, and iteratively find the lowest OML level

        for practice in practices:
            practice_id = practice[0]
            kpi = DBHelper.run_query_flat_results("SELECT ID,NAME FROM KPI WHERE PRACTICE_ID=%s" % practice_id)
            kpi_id = kpi[0]
            kpi_name = kpi[1]
            practice_rollup_bu = DBHelper.run_query_flat_results("SELECT VALUE1,VALUE2,COMPUTED_VALUE,OML_LEVEL "
                                                                 "FROM METRICS_SUMMARY "
                                                                 "WHERE ROLLUP_ID=2 AND KPI_ID=%s "
                                                                 "AND OML_BU_ID=%s %s"
                                                                 "ORDER BY INGEST_BATCH DESC LIMIT 1"
                                                                 % (kpi_id, bu_id, date_clause))
            if len(practice_rollup_bu) > 0:
                if kpi_id in KPIS_ENABLED_FOR_SUMMARY:
                    bu_oml_level = min(bu_oml_level, practice_rollup_bu[3])
                    print("%s: %d/%d (%0.2f%%)" % (kpi_name, practice_rollup_bu[0], practice_rollup_bu[1],
                                                   100*practice_rollup_bu[2]))
                    update_column_values(header_col, kpi_name, data_col, "%d/%d (%0.2f%%)"
                               % (practice_rollup_bu[0], practice_rollup_bu[1], 100*practice_rollup_bu[2]))
                else:
                    print("%s: N/A" % kpi_name)
                    update_column_values(header_col, kpi_name, data_col, "N/A")
            else:
                print("%s: N/A" % kpi_name)
                update_column_values(header_col, kpi_name, data_col, "N/A")

        print("OML Level: %s" % (bu_oml_level if 0 <= bu_oml_level <= 5 else 0 if bu_oml_level> 5 else "N/A"))
        update_column_values(header_col, "OML Level", data_col, bu_oml_level if 0 <= bu_oml_level <= 5 else 0 if bu_oml_level> 5 else "N/A")

        teams = DBHelper.run_query("SELECT ID,NAME FROM OML_TEAM WHERE OML_BU_ID=%s ORDER BY NAME" % bu_id)
        for team in teams:
            team_oml_level = MAX_LEVEL  # Start with Max level, and iteratively find the lowest OML level
            team_id = team[0]
            team_name = team[1]
            print(" ")
            update_column_values(header_col, '', data_col, '')
            print(team_name)
            update_column_values(header_col, team_name, data_col, '')
            for practice in practices:
                practice_id = practice[0]
                kpi = DBHelper.run_query_flat_results("SELECT ID,NAME FROM KPI WHERE PRACTICE_ID=%s" % practice_id)
                kpi_id = kpi[0]
                kpi_name = kpi[1]
                practice_rollup_team = DBHelper.run_query_flat_results("SELECT VALUE1,VALUE2,COMPUTED_VALUE,OML_LEVEL "
                                                                       "FROM METRICS_SUMMARY "
                                                                       "WHERE ROLLUP_ID=3 AND KPI_ID=%s "
                                                                       "AND OML_BU_ID=%s AND OML_TEAM_ID=%s %s "
                                                                       "ORDER BY INGEST_BATCH DESC LIMIT 1"
                                                                       % (kpi_id, bu_id, team_id, date_clause))
                if len(practice_rollup_team) > 0:
                    if kpi_id in KPIS_ENABLED_FOR_SUMMARY:
                        team_oml_level = min(team_oml_level, practice_rollup_team[3])
                        print("%s: %d/%d (%0.2f%%)" % (
                        kpi_name, practice_rollup_team[0], practice_rollup_team[1], 100 * practice_rollup_team[2]))
                        update_column_values(header_col, kpi_name, data_col,
                                             "%d/%d (%0.2f%%)" % (practice_rollup_team[0], practice_rollup_team[1],
                                                                  100 * practice_rollup_team[2]))
                    else:
                        print("%s: N/A" % kpi_name)
                        update_column_values(header_col, kpi_name, data_col, "N/A")
                else:
                    print("%s: N/A" % kpi_name)
                    update_column_values(header_col, kpi_name, data_col, "N/A")

            print("OML Level: %s" % (team_oml_level if 0 <= team_oml_level <= 5 else 0 if team_oml_level> 5 else "N/A"))
            update_column_values(header_col, "OML Level", data_col, team_oml_level if 0 <= team_oml_level <= 5 else 0 if team_oml_level> 5 else "N/A")

    if args.output == "plain":
        # we're done
        return

    # Update the actual google spreadsheet with the data collected above
    if args.rollup == "twitch":
        worksheet = get_worksheet("Twitch")
    else:
        worksheet = get_worksheet(args.bu)

    worksheet.update_col(1, header_col)
    data_col_num = week_of_year - WEEK_OFFSET + FIRST_DATA_COL
    for attempt in range(0, 3):
        try:
            worksheet.update_col(data_col_num, data_col)
            break
        except Exception:
            time.sleep(BACKOFF_RETRY_WAIT * attempt)  # simple backoff

    # boldface some header cells
    for row in range(1, len(header_col), 13):
        row1 = row
        row2 = row1+11
        for attempt in range(0, 3):
            try:
                worksheet.cell("A%d" % row1).set_text_format('bold', True)
                worksheet.cell("A%d" % row2).set_text_format('bold', True)
                break
            except Exception:
                time.sleep(BACKOFF_RETRY_WAIT * attempt)  # simple backoff


def main():
    # handler for CLI

    # declare all the CLI arguments
    cmds = ['summary']  # Add more as needed
    ref_bu_set = OrgHelper.ref_BUs()
    ref_team_set = OrgHelper.ref_teams()

    # setup the CLI parser with all the CLI arguments
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(dest='command')
    for cmd in cmds:
        sub = subparsers.add_parser(cmd, help="query OML metrics: %s" % cmd)
        if cmd == "summary":
            sub.add_argument("--rollup", choices=['twitch', 'bu'], required=True)
            sub.add_argument("--bu", choices=list(ref_bu_set), required=False)
            sub.add_argument("--date", required=False, help="Summary date in YYYY/MM/DD format")

        sub.add_argument("--output", choices=['plain', 'google'], required=False, default='plain')
        sub.set_defaults(func=eval("output_%s" % cmd.replace("-", "_")))  # map metric to handler function ( output_<name> )
    args = parser.parse_args()
    args.func(args)

    return


if __name__ == '__main__':
    # entrypoint for CLI
    main()
