import argparse
import csv

import os

import requests
from alembic import context, op
from alembic.config import Config
from sqlalchemy import engine_from_config, pool

# add your model's MetaData object here
# for 'autogenerate' support
# from myapp import mymodel
# target_metadata = mymodel.Base.metadata
target_metadata = None

from datasource import OMLCapacity


def add_jira_category(conn, jira_category, oml_bu):
    entry = conn.execute(
        "SELECT * FROM JIRA_CATEGORY JC, OML_BU OB  WHERE JC.NAME='%s' AND JC.OML_BU_ID=OB.ID AND OB.NAME='%s'" %
        (jira_category, oml_bu)).fetchone()
    if entry is None:
        conn.execute(
            "INSERT INTO JIRA_CATEGORY(NAME, OML_BU_ID) SELECT '%s' AS NAME, `ID` FROM OML_BU WHERE NAME='%s'" %
            (jira_category, oml_bu))


def add_incident_org(conn, org, oml_bu):
    entry = conn.execute(
        "SELECT * FROM INCIDENT_ORG IO, OML_BU OB WHERE IO.NAME='%s' AND IO.OML_BU_ID=OB.ID AND OB.NAME='%s'" %
        (org, oml_bu)).fetchone()
    if entry is None:
        conn.execute(
            "INSERT INTO INCIDENT_ORG(NAME, OML_BU_ID) SELECT '%s' AS NAME, `ID` FROM OML_BU WHERE NAME='%s'" %
            (org, oml_bu))


def populate_omldata_tables(conn, input_csv):
    with open(input_csv) as f:
        reader = csv.reader(f)
        count = 0

        for row in reader:
            o_bu, o_team, j_project, j_key, o_services, repo, j_inc_components, lead = row
            if count == 0:  # skip header row
                count += 1
                continue

            # add to OML_BU
            oml_bu = conn.execute("SELECT * FROM OML_BU WHERE NAME='%s'" % o_bu).fetchone()
            if oml_bu is None:
                conn.execute("INSERT INTO OML_BU(NAME) VALUES('%s')" % o_bu)
                oml_bu = conn.execute("SELECT * FROM OML_BU WHERE NAME='%s'" % o_bu).fetchone()
                print("Added OML BU: %s" % o_bu)

            # add to OML_TEAM
            oml_team = conn.execute(
                "SELECT * FROM OML_TEAM WHERE OML_BU_ID=%s AND NAME='%s'" % (oml_bu[0], o_team)).fetchone()
            if oml_team is None:
                conn.execute("INSERT INTO OML_TEAM(NAME,OML_BU_ID,ENG_LEAD) VALUES('%s', %s, '%s')" %
                             (o_team, oml_bu[0], lead,))
                oml_team = conn.execute(
                    "SELECT * FROM OML_TEAM WHERE OML_BU_ID=%s AND NAME='%s'" % (oml_bu[0], o_team)).fetchone()
                print("Added OML Team: %s" % o_team)

            # add to OML_SERVICE (1-to-many)
            for service in o_services.split("\n"):
                if len(service) > 0:
                    oml_service = conn.execute("SELECT * FROM OML_SERVICE WHERE OML_TEAM_ID=%s AND NAME='%s'"
                                               % (oml_team[0], service)).fetchone()
                    if oml_service is None:
                        conn.execute(
                            "INSERT INTO OML_SERVICE(NAME,OML_TEAM_ID) VALUES('%s', %s)" % (service, oml_team[0]))
                        print("Added OML Service: %s" % service)

            # add to jira_project
            if len(j_project) > 0 and len(j_key) > 0:
                jira_project = conn.execute("SELECT * FROM JIRA_PROJECT WHERE `KEY`='%s'" % j_key).fetchone()
                if jira_project is None:
                    conn.execute(
                        "INSERT INTO JIRA_PROJECT(`NAME`, `KEY`, `OML_TEAM_ID`) SELECT '%s' AS NAME, '%s' AS `KEY`, `ID` FROM OML_TEAM WHERE NAME='%s'" %
                        (j_project, j_key, o_team))
                    print("Added JIRA Project: %s" % j_project)
                else:
                    print("WARNING: Entry for KEY=%s already exists in JIRA Project" % j_key)

            # add ghe_repos
            for full_repo_name in repo.split("\n"):
                if len(full_repo_name) > 0:
                    ghe_repo = conn.execute(
                        "SELECT * FROM GHE_REPO GR, OML_TEAM OT WHERE GR.NAME='%s' AND GR.OML_TEAM_ID=OT.ID AND OT.NAME='%s'" %
                        (full_repo_name.lower(), o_team)).fetchone()
                    if ghe_repo is None:
                        conn.execute(
                            "INSERT INTO GHE_REPO(NAME, OML_TEAM_ID) SELECT '%s' AS NAME, `ID` FROM OML_TEAM WHERE NAME='%s'" %
                            (full_repo_name.lower(), o_team))
                        print("Added GHE Repo: %s" % full_repo_name.lower())

            # add to incident component
            for inc_component in j_inc_components.split("\n"):
                if len(inc_component) > 0:
                    inc_component_result = conn.execute(
                        "SELECT * FROM INCIDENT_COMP IC, OML_TEAM OT WHERE IC.NAME='%s' AND IC.OML_TEAM_ID=OT.ID AND OT.NAME='%s'" %
                        (inc_component, o_team)).fetchone()
                    if inc_component_result is None:
                        inc_component_name_result = conn.execute("SELECT * FROM INCIDENT_COMP IC WHERE IC.NAME='%s'"
                                                                 % inc_component).fetchone()
                        if inc_component_name_result is None:
                            conn.execute(
                                "INSERT INTO INCIDENT_COMP(NAME, OML_TEAM_ID) SELECT '%s' AS NAME, `ID` FROM OML_TEAM WHERE NAME='%s'" %
                                (inc_component, o_team))
                            print("Added Incident Component: %s" % inc_component)
                        else:
                            print("WARNING: Incident Component already exists: %s" % inc_component_name_result)
            count += 1

    # static - add to jira_category
    add_jira_category(conn, "Broadcaster Success", "Broadcaster")
    add_jira_category(conn, "CAPE", "CAPE")
    add_jira_category(conn, "Commerce", "Commerce")
    add_jira_category(conn, "Community", "Community")
    add_jira_category(conn, "Developer Success", "Dev Success")

    # static - add to incident org
    add_incident_org(conn, "Other-BS", "Broadcaster")
    add_incident_org(conn, "Other-CAPE", "CAPE")
    add_incident_org(conn, "Commerce", "Commerce")
    add_incident_org(conn, "Community", "Community")
    add_incident_org(conn, "Other-DS", "Dev Success")


def add_catalog_bu(conn, catalog_bu, ext_key, oml_bu):
    entry = conn.execute(
        "SELECT * FROM STATUS_BU SB, OML_BU OB WHERE SB.NAME='%s' AND SB.EXT_KEY='%s' AND SB.OML_BU_ID=OB.ID AND OB.NAME='%s'" %
        (catalog_bu, ext_key, oml_bu)).fetchone()
    if entry is None:
        conn.execute(
            "INSERT INTO STATUS_BU(NAME, EXT_KEY, OML_BU_ID) SELECT '%s' AS NAME, '%s' as EXT_KEY, `ID` FROM OML_BU WHERE NAME='%s'" %
            (catalog_bu, ext_key, oml_bu))
        print("Added Status BU %s" % entry)


def lookup(haystack, index_or_key_name, needle, patch):
    for entry in haystack:
        if patch:
            if entry[index_or_key_name].replace(" ", "").lower().strip() == needle.replace(" ", "").lower().strip() or entry[index_or_key_name].startswith(needle) or needle.startswith(entry[index_or_key_name]):
                return entry
        else:
            if entry[index_or_key_name] == needle:
                return entry
    return None


def populate_catalog_tables(conn, input_csv):
    # static - add to catalog bu
    add_catalog_bu(conn, "Broadcaster", "BU_BROADCASTER", "Broadcaster")
    add_catalog_bu(conn, "CAPE", "BU_CAPE", "CAPE")
    add_catalog_bu(conn, "Commerce", "BU_COMMERCE", "Commerce")
    add_catalog_bu(conn, "Community", "BU_COMMUNITY", "Community")
    add_catalog_bu(conn, "Developer Success", "BU_DEVELOPER", "Dev Success")

    orgs_known_to_service_catalog = requests.get("https://status.internal.justin.tv/api/v1/orgs/").json()

    # Populate service catalog teams.
    # Use csv from reference google spreadsheet to map OML team <-> service catalog team, as this mapping is
    # critical for team level report integrity and needs human vetting.\
    oml_to_status_team_mapping = []
    with open(input_csv) as f:
        reader = csv.reader(f)
        count = 0

        teams_known_to_service_catalog = requests.get("https://status.internal.justin.tv/api/v1/teams/").json()
        services_known_to_service_catalog = requests.get("https://status.internal.justin.tv/api/v1/services/").json()
        services_known_to_oml = conn.execute("SELECT * FROM OML_SERVICE").fetchall()

        for row in reader:
            if count == 0:  # skip header row
                count += 1
                continue

            oml_bu, oml_team, catalog_team, match = row

            if len(catalog_team) <= 1:
                continue

            oml_to_status_team_mapping.append(row)
            # check csv entry against persisted entries in service catalog
            for team_entry in teams_known_to_service_catalog:
                if team_entry['name'] == catalog_team:  # matching team name
                    team_ext_id = team_entry['id']
                    has_catalog_to_oml_team_mapping = len(oml_team) > 1
                    if has_catalog_to_oml_team_mapping:
                        test = conn.execute("SELECT * FROM STATUS_TEAM WHERE `NAME`='%s' AND EXT_ID='%s'"
                                            % (catalog_team, team_ext_id)).fetchone()
                        if test is None:
                            conn.execute("INSERT INTO STATUS_TEAM(NAME, EXT_ID, OML_TEAM_ID) "
                                         "SELECT '%s', '%s', ID FROM OML_TEAM OT WHERE OT.NAME='%s'"
                                         % (catalog_team, team_ext_id, oml_team))
                            print("Added Catalog Team:OML Team Mapping %s:%s" % (catalog_team, oml_team))
                    else:
                        test = conn.execute("SELECT * FROM STATUS_TEAM WHERE `NAME`='%s' AND EXT_ID='%s'"
                                            % (catalog_team, team_ext_id)).fetchone()
                        if test is None:
                            conn.execute("INSERT INTO STATUS_TEAM(NAME, EXT_ID) "
                                         "SELECT '%s', '%s'" % (catalog_team, team_ext_id))
                            print("Added Catalog Team %s that's NOT mapped to any OML Team" % catalog_team)

    # Next, populate services (from service catalog's point of view)
    for svc_entry in services_known_to_service_catalog:
        catalog_service_name = svc_entry['name']
        catalog_service_ext_id = svc_entry['id']
        catalog_service_team_id = svc_entry['team']
        print("Processing %s" % catalog_service_name)

        db_entry = conn.execute("SELECT ID FROM STATUS_SERVICE WHERE `NAME`='%s' AND EXT_ID='%s'"
                                % (catalog_service_name, catalog_service_ext_id)).fetchall()
        if len(db_entry) == 0:
            # Not found in STATUS_SERVICE

            # First, add to it and OML_SERVICE
            oml_service_from_db = conn.execute("SELECT `NAME`, `ID` FROM OML_SERVICE").fetchall()
            oml_service_name_id = lookup(oml_service_from_db, 0, catalog_service_name, True)  # Do fuzzy lookup

            if oml_service_name_id is None:
                # Not found in OML_SERVICE, so add it

                # Get OML TEAM ID, by analyzing CATALOG TEAM
                db_entry = conn.execute("SELECT OML_TEAM_ID FROM STATUS_TEAM WHERE EXT_ID='%s'"
                                        % catalog_service_team_id).fetchone()
                if db_entry is None or db_entry[0] is None:
                    print("WARNING: OML_TEAM does not have an equivalent of STATUS_TEAM: %s. "
                          "Update the catalog mapping CSV!" % catalog_service_name)
                    continue

                print("Adding %s to OML_SERVICE" % catalog_service_name)
                conn.execute("INSERT INTO OML_SERVICE(`NAME`, OML_TEAM_ID) VALUES('%s', %s)"
                             % (catalog_service_name, db_entry[0]))
                oml_service_id = conn.execute("SELECT ID FROM OML_SERVICE WHERE `NAME`='%s' AND OML_TEAM_ID=%s"
                                        % (catalog_service_name, db_entry[0])).fetchone()[0]
                oml_service_name = catalog_service_name
            else:
                oml_service_name = oml_service_name_id[0]
                oml_service_id = oml_service_name_id[1]


            # Then, add to STATUS_SERVICE
            oml_team_id = conn.execute("SELECT `OML_TEAM_ID` FROM OML_SERVICE WHERE ID=%s"
                                       % oml_service_id).fetchone()[0]
            print("Adding %s to STATUS_SERVICE" % catalog_service_name)
            conn.execute(
                "INSERT INTO STATUS_SERVICE(NAME, EXT_ID, OML_SERVICE_ID, OML_TEAM_ID) "
                "VALUES('%s', %s, %s, %s)"
                % (catalog_service_name, catalog_service_ext_id, oml_service_id, oml_team_id))
            print("Added Mapping Catalog Service:OML Service %s:%s"
                  % (catalog_service_name, oml_service_name))
        else:
            # Found in STATUS_SERVICE
            # Update it in STATUS_SERVICE AND OML_SERVICE if necessary
            pass




def add_capacity_bu(conn, capacity_bu, ext_key, oml_bu):
    entry = conn.execute("SELECT * FROM CAPACITY_BU CB, OML_BU OB "
                         "WHERE CB.NAME='%s' AND CB.EXT_KEY='%s' AND CB.OML_BU_ID=OB.ID AND OB.NAME='%s'"
                         % (capacity_bu, ext_key, oml_bu)).fetchone()
    if entry is None:
        conn.execute("INSERT INTO CAPACITY_BU(NAME, EXT_KEY, OML_BU_ID) "
                     "SELECT '%s' AS NAME, '%s' AS EXT_KEY, `ID` FROM OML_BU WHERE NAME='%s'"
                     % (capacity_bu, ext_key, oml_bu))


def populate_capacity_tables(conn, input_csv):
    # static - add to capacity bu
    add_capacity_bu(conn, "Broadcaster Success", "BU_BROADCASTER", "Broadcaster")
    add_capacity_bu(conn, "CAPE", "BU_CAPE", "CAPE")
    add_capacity_bu(conn, "Commerce", "BU_COMMERCE", "Commerce")
    add_capacity_bu(conn, "Community", "BU_COMMUNITY", "Community")
    add_capacity_bu(conn, "Developer Success", "BU_DEVELOPER", "Dev Success")

    # get capacity group -> capacity BU mapping temporarily
    cap_group_bu_map = {}
    cap_data = OMLCapacity().get_config_management_status_summary("bu")
    for entry in cap_data:
        entry_id = entry['id']
        if entry_id.startswith("GRP_"):
            cap_group_bu_map[entry_id] = entry['parent_id']

    # Capacity has an extra level ("Group") in the app's org stricture (BU -> Group -> Business Function)
    # Flatten it to be more similar to all other app's org structure (BU -> Team)
    for entry in OMLCapacity().get_config_management_status_summary("bu"):
        # TODO: Add mapping to oml_team_id
        entry_id = entry['id']
        if not entry_id.startswith("BU_"):
            parent_bu_id = cap_group_bu_map.get(entry['parent_id'])
            if parent_bu_id is not None:
                test = conn.execute("SELECT * FROM CAPACITY_GROUP WHERE `NAME`='%s' AND EXT_KEY='%s'"
                                    % (entry['name'], entry_id)).fetchone()
                if test is None:
                    conn.execute("INSERT INTO CAPACITY_GROUP(NAME,EXT_KEY,CAPACITY_BU_ID) "
                                 "SELECT '%s', '%s', CB.ID FROM CAPACITY_BU CB WHERE CB.EXT_KEY='%s'"
                                 % (entry['name'], entry_id, parent_bu_id))
                    print("Added Capacity Team: %s / %s (%s)" % (parent_bu_id, entry['name'], entry_id))


EXT_SYSTEMS = ['omldata', 'catalog', 'capacity']


def setup_db_connection(alembic_env):
    alembic_config = Config("alembic.ini")
    db_config = alembic_config.get_section(alembic_env)
    conf_dict = {}
    url = db_config['sqlalchemy.url']
    if '${ALEMBIC_USER}' in url:
        url = url.replace('${ALEMBIC_USER}', os.environ['ALEMBIC_USER'])
    if '${ALEMBIC_PASSWORD}' in url:
        url = url.replace('${ALEMBIC_PASSWORD}', os.environ['ALEMBIC_PASSWORD'])

    conf_dict['sqlalchemy.url'] = url

    engine = engine_from_config(
        conf_dict,
        prefix='sqlalchemy.',
        poolclass=pool.NullPool)
    conn = engine.connect()

    return conn


def main():
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(dest='command')
    for ext_system in EXT_SYSTEMS:
        sub = subparsers.add_parser(ext_system, help="query OML metrics: %s" % ext_system)
        sub.set_defaults(func=eval("populate_%s_tables" % ext_system))  # map cli to python function
        sub.add_argument('--alembic-env', choices=['development', 'staging', 'production'], required=True,
                         help="Alembic env (referenced in alembic.ini)")
        sub.add_argument('--input-csv', help="Input CSV file, eg oml-jira-ghe-09292017.csv or ")

    args = parser.parse_args()
    conn = setup_db_connection(args.alembic_env)
    args.func(conn, args.input_csv)


if __name__ == "__main__":
    main()