import functools

import datetime
import jira.client
import json

import os

import multiprocessing
import requests
import time

from pathlib import Path

from multiprochelper import run_parallel
from sandstormhelper import SandStormHelper


class OMLHTTPBase():
    def __init__(self, base_url):
        self.base_url = base_url

    def get(self, api_path, headers, params, timeout):
        resp = None
        full_url = "%s%s" % (self.base_url, api_path)
        for retry in range(0, 3):
            try:
                resp = requests.get(full_url, headers=headers, params=params, timeout=timeout)
                break
            except Exception as e:
                print("ERROR: API call to %s failed. Will retry as needed. Error =  %s" % (full_url, e))
                time.sleep(5 * retry)

        if resp is None:
            return None

        if resp.status_code != 200:
            raise Exception("ERROR: Could not get valid response for %s after all retries. Last status=%s"
                            % (full_url, resp.status_code))

        return resp.json()


class OMLJira(object):
    """
    OML Jira class queries jira for OML. All inputs to these queries must be in jira-scope (ie, do not pass
    Internal Status team names to Jira)

    """
    def __init__(self):
        # at command line, run this to get tokens:
        # jirashell -pt -s 'https://jira.twitch.com' -u "yourusername" -p "whateverpassword" -od -ck "yourclientkeyhere" -k private.pem

        self.client = self.init_jira()
        with open('config/jira_queries.json', 'r') as jira_queries_file:
            self.jira_queries = json.load(jira_queries_file)

    @functools.lru_cache()
    def init_jira(self):
        """
        Initializes jira client. REQUIRES 2 files in working directory:
        1) privkey.pem
        2) jira_token.auth

        :return: jira client object
        """
        jira_options = {'server': "https://jira.twitch.com"}
        jira_key_location = 'privkey.pem'
        jira_token_location = 'jira_token.auth'

        jira_key_file = Path(jira_key_location)
        jira_token_file = Path(jira_token_location)
        if jira_key_file.exists() and jira_token_file.exists():
            with open(jira_key_location, 'r') as key_cert_file:
                key_cert_data = key_cert_file.read()
            with open(jira_token_location, 'r') as credentials_file:
                tokens = credentials_file.readline()
        else:
            key_cert_data = SandStormHelper.get_secrets_str('qa-eng/oml/%s/jira-privkey.pem' % os.getenv('environment', None))
            tokens = SandStormHelper.get_secrets_str('qa-eng/oml/%s/jira_token.auth' % os.getenv('environment', None))

        if tokens is None or key_cert_data is None:
            raise Exception("ERROR: Could not get jira secrets from file system or sandstorm")

        access_token = tokens.split(':')[0]
        access_token_secret = tokens.split(':')[1]

        oauth_dict = {
            'access_token': access_token,
            'access_token_secret': access_token_secret,
            'consumer_key': 'qe',
            'key_cert': key_cert_data
        }

        try:
            client = jira.client.JIRA(options=jira_options, oauth=oauth_dict, validate=True, logging=False)
        except Exception as e:
            print("WARNING: Cannot connect to JIRA: {0}".format(str(e)))
            raise e

        return client

    def run_named_query(self, query_name, rollup, **kwargs):
        """
        Runs a named query declared in config/jira_queries.json

        :param query_name: query name
        :param rollup: rollup
        :param kwargs: parameters to query
        :return:
        """
        if rollup is None:
            raw_query = self.jira_queries[query_name]
        else:
            raw_query = self.jira_queries[query_name][rollup]

        resolved_query = raw_query  # give resolved_query a default value, as raw_query may NOT need to be resolved

        for key, value in kwargs.items():
            if type(value) != list:
                resolved_query = raw_query.replace(":%s" % key, "\"%s\"" % value)
            elif type(value) == list and len(value) == 1:
                resolved_query = raw_query.replace(":%s" % key, "\"%s\"" % value[0])
            else:
                combined_values = []
                for each_value in value:
                    combined_values.append("\"%s\"" % each_value)
                resolved_query = raw_query.replace(":%s" % key, ",".join(combined_values))
        return self.client.search_issues(resolved_query, maxResults=1000, fields=[])

    def get_RIs_created(self, rollup, bu, team=None):
        """
        Queries RIs created

        :param rollup: rollup (bu or team)
        :param bu: BU name ("Category", in jira terms)
        :param team: Team name ("Project"), in jira terms)
        :return: RIs created
        """

        if rollup == "bu":
            results = self.run_named_query("All RIs are closed within SLA - created", "bu", name=bu)
        elif rollup == "team":
            results = self.run_named_query("All RIs are closed within SLA - created", "team", name=team)

        return results

    def get_RIs_resolved(self, rollup, bu, team=None):
        """
        Queries RIs resolved

        :param rollup: rollup (bu or team)
        :param bu: BU name ("Category", in jira terms)
        :param team: Team name ("Project"), in jira terms)
        :return: RIs resolved
        """
        if rollup == "bu":
            # roll-up is business unit level
            results = self.run_named_query("All RIs are closed within SLA - resolved", "bu", name=bu)
        elif rollup == "team":
            results = self.run_named_query("All RIs are closed within SLA - resolved", "team", name=team)

        return results


    # OML: The goal is to keep the close rate of RIs close to the open rate -- within 10% i.e. 90% of cumulative RIs are closed.
    def get_percent_RIs_resolved_in_timely_manner(self, rollup, bu, team=None):
        created = self.get_RIs_created(rollup, bu, team)
        resolved = self.get_RIs_resolved(rollup, bu, team)
        return len(created), len(resolved)


    def get_sev123_prod_bugs_created(self, rollup, category, projects=None):
        """
        Queries sev123_prod_bugs created

        :param rollup: rollup (bu or team)
        :param category: "Category", in jira terms)
        :param projects: "Project", in jira terms)
        :return: sev123_prod_bugs created
        """

        if rollup == "bu":
            results = self.run_named_query("All Sev-1 through Sev-3 bugs should be resolved in a timely manner - created"
                                           , "bu", name=category)
        elif rollup == "team":
            results = self.run_named_query("All Sev-1 through Sev-3 bugs should be resolved in a timely manner - created"
                                           , "team", name=projects)

        return results

    def get_sev123_prod_bugs_resolved(self, rollup, category, projects=None):
        """
        Queries sev123_prod_bugs resolved

        :param rollup: rollup (bu or team)
        :param category: BU name ("Category", in jira terms)
        :param projects: Team name ("Project"), in jira terms)
        :return: sev123_prod_bugs resolved
        """
        if rollup == "bu":
            # roll-up is business unit level
            results = self.run_named_query("All Sev-1 through Sev-3 bugs should be resolved in a timely manner - resolved",
                                           "bu", name=category)
        elif rollup == "team":
            results = self.run_named_query("All Sev-1 through Sev-3 bugs should be resolved in a timely manner - resolved",
                                           "team", name=projects)

        return results


    # OML: The goal is to keep the close rate of sev123_prod_bugs close to the open rate -- within 10% i.e. 90% of cumulative sev123_prod_bugs are closed.
    def get_percent_sev123_prod_bugs_resolved_in_timely_manner(self, rollup, category, projects=None):
        created = self.get_sev123_prod_bugs_created(rollup, category, projects)
        resolved = self.get_sev123_prod_bugs_resolved(rollup, category, projects)
        return len(created), len(resolved)


    def get_incidents(self, rollup, org):
        if rollup == "bu" and org is not None:
            return self.run_named_query("All incidents", "bu", name=org)
        elif rollup == "twitch" or rollup == "all":
            return self.run_named_query("All incidents", "all")

        return None


class OMLInternalStatus(OMLHTTPBase):
    """
    OML Internal Status class to query OML metrics from Internal Status app

    All inputs must be in Internal Status values. Outputs will be in raw Internal Status values as well.
    """
    BASE_URL = "https://hms-esk.internal.justin.tv/v1"

    def __init__(self):
        super().__init__(OMLInternalStatus.BASE_URL)

    @functools.lru_cache()
    def get_availability_report(self):
        # Our reporting time frame syncs with finance e.g. SLA instances (Monday-Sunday).
        # Note, this report api returns the entire report, so there's no way to reduce the size of the query
        params = {'reportID': 'availability'}
        resp_json = self.get("/reportitems/", None, params, 5)

        # Reporting Window follows Finance Window (Monday - Sunday)
        reporting_window_start_date = resp_json[-1]['date']
        availability_data = resp_json[-1]['data']

        return reporting_window_start_date, availability_data

    # OML:  All services operate within an availability SLA
    @functools.lru_cache()
    def get_services_operate_within_an_availability_SLA(self, rollup, status_bu, status_team=None, service=None):
        reporting_window_start_date, availability_data = self.get_availability_report()
        results = []
        for entry in availability_data:
            entry_bu = entry['Name']
            if 'ServiceReports' in entry:
                for report in entry['ServiceReports']:
                    team = report['TeamName']
                    if (status_bu == '*' or status_bu == entry_bu) and (status_team == '*' or status_team == team):
                        if 'Availability' in report:
                            service_name = report['Name']
                            results.append((entry_bu, team, service_name, report['SLA'], report['Availability']))
                            # do something

        # return reporting_window_start_date, list of tuples (bu, team, SLA, availability)
        return reporting_window_start_date, results


class OMLJenkins(object):
    pass


class OMLCodeCov(OMLHTTPBase):
    """
    OMLCodeCov queries OML metrics from Codecov server

    API Reference: https://docs.codecov.io/v4.3.6/reference

    """

    BASE_URL = "https://codecov.internal.justin.tv/api"

    def __init__(self, api_token=None):
        super().__init__(OMLCodeCov.BASE_URL)
        if api_token is not None:
            self.api_token = api_token
        else:
            env_api_token = os.getenv("CODECOV_API_TOKEN")
            if env_api_token is None:
                env_api_token = SandStormHelper.get_secrets_str('qa-eng/oml/%s/codecov' % os.environ['environment'])

            if env_api_token is None:
                raise Exception("ERROR: Environment var CODECOV_API_TOKEN is not set, "
                                "or unable to retrieve qa-eng/oml/staging/codecov from Sandstorm. "
                                "Aborting codecov initialization")

            self.api_token = env_api_token

    def get_repo_codecov(self, full_repo_name):
        auth = {'Authorization': 'token %s' % self.api_token}
        resp_json = None
        try:
            resp_json = self.get("/ghe/%s" % full_repo_name, auth, None, 5)
        except Exception as e:
            print("ERROR: %s" % e)

        if resp_json is None:
            return None

        if 'commit' in resp_json and 'totals' in resp_json['commit'] and resp_json['commit']['totals'] is not None:
            coverage_info = resp_json['commit']['totals']
            # Reference: https://docs.codecov.io/v4.3.6/reference#totals
            return {'repo': full_repo_name, 'h': coverage_info['h'], 'n': coverage_info['n']}
        else:
            return None

    def get_builds_codecov(self, full_repo_name):
        auth = {'Authorization': 'token %s' % self.api_token}
        params = {'method': 'max', 'agg': 'day', 'time': '7d', 'inc': 'totals'}
        resp_json = None
        try:
            resp_json = self.get("/ghe/%s/graphs/commits.json" % full_repo_name, auth, params, 5)
        except Exception as e:
            print("ERROR: %s" % e)

        return resp_json


class OMLCapacity(OMLHTTPBase):
    """
    OMLCapacity queries capacity dashboard app

    """

    BASE_URL = "http://dashboard.capacity.internal.justin.tv"

    def __init__(self):
        super().__init__(OMLCapacity.BASE_URL)

    def get_config_management_status_summary(self, rollup):
        resp_json = self.get("/inventory/config_mgmt/json/df_org/", None, None, 5)
        return resp_json['df_org']


class OMLSkadi(OMLHTTPBase):
    BASE_URL = "http://clean-deploy.internal.justin.tv"

    def __init__(self, api_token=None):
        super().__init__(OMLSkadi.BASE_URL)
        if api_token is not None:
            self.api_token = api_token
        else:
            env_api_token = os.getenv("GITHUB_ACCESS_TOKEN")
            if env_api_token is None:
                env_api_token = SandStormHelper.get_secrets_str('qa-eng/oml/%s/ghe' % os.getenv('environment', None))

            self.api_token = env_api_token
            if env_api_token is None:
                raise Exception("ERROR: Environment var GITHUB_ACCESS_TOKEN is not set. Aborting Skadi client init")

    def get_environments(self, env_name=None):
        return self.get("/v1/environments", {'GithubAccessToken': self.api_token}, {'per_page': '1000000'}, 5)


class OMLRPS(OMLHTTPBase):
    BASE_URL = "http://rockpaperscissors.internal.justin.tv/api/v1"

    SECONDS_PER_BLOCK = 2 * 60 * 60
    NUM_BLOCKS_IN_7_DAYS = 12 * 7

    # HEADERS = {'Accept': 'application/json', 'Accept-Encoding' : 'gzip, deflate'}
    HEADERS = {'Accept': 'application/json' }

    def __init__(self):
        super().__init__(OMLRPS.BASE_URL)

    def get_wrapper(self, path, headers, params, timeout):
        resp_json = None
        try:
            resp_json = self.get(path, headers, params, timeout)
        except Exception as e:
            print("ERROR: failed to invoke %s" % path)

        return resp_json

    def get_repos_with_test_results(self):
        repos = set()
        end_time = int(time.time())  # timestamp in seconds

        invocation_params_list = []
        for i in range(1, OMLRPS.NUM_BLOCKS_IN_7_DAYS):  # go back some days
            start_time = int(end_time - OMLRPS.SECONDS_PER_BLOCK)
            print("Will query for time frame %s - %s" % (datetime.datetime.fromtimestamp(start_time),
                                                         datetime.datetime.fromtimestamp(end_time)))
            param_tuple = ("/events/types/JenkinsTestResults/%d/%d" % (start_time, end_time), OMLRPS.HEADERS, None, 5)
            invocation_params_list.append(param_tuple)
            end_time = start_time

        all_resp_json = run_parallel(self.get_wrapper, invocation_params_list, 6)

        for resp_json in all_resp_json:
            if resp_json is None:
                continue

            if 'events' not in resp_json:
                continue

            for event in resp_json['events']:
                for attribute in event['attributes']:
                    if attribute['key'] == 'github_repository':
                        repo_url = attribute['value']
                        repo_slug = '/'.join(repo_url.split('/')[-2:])
                        repos.add(repo_slug)

            #print(repos)

        return list(repos)

    def get_builds_with_test_results(self):
        """
        Query RPS for builds that are considered test builds (builds that run tests)

        :return: A dictionary of job name to number of test build
        """
        jobs_to_build_count = {}
        end_time = int(time.time())

        invocation_params_list = []
        for i in range(1, OMLRPS.NUM_BLOCKS_IN_7_DAYS):  # go back some days
            start_time = int(end_time - OMLRPS.SECONDS_PER_BLOCK)
            print("Will query for time frame %s - %s" % (datetime.datetime.fromtimestamp(start_time),
                                                         datetime.datetime.fromtimestamp(end_time)))
            param_tuple = ("/events/types/JenkinsTestResults/%d/%d" % (start_time, end_time), OMLRPS.HEADERS, None, 5)
            invocation_params_list.append(param_tuple)
            end_time = start_time

        all_resp_json = run_parallel(self.get_wrapper, invocation_params_list, 6)

        for resp_json in all_resp_json:
            if resp_json is None:
                continue

            if 'events' not in resp_json:
                continue

            for event in resp_json['events']:
                for attribute in event['attributes']:
                    if attribute['key'] == 'jenkins_job_name':
                        job_name = attribute['value']

                        if job_name in jobs_to_build_count:
                            jobs_to_build_count[job_name] = jobs_to_build_count[job_name] + 1
                        else:
                            jobs_to_build_count[job_name] = 1

        return jobs_to_build_count


class OMLGraphite(OMLHTTPBase):
    BASE_URL = "https://grafana.internal.justin.tv/api/datasources/proxy/2"
    MAX_DATA_POINTS_PER_DAY = 24

    def __init__(self):
        super().__init__(OMLGraphite.BASE_URL)

    def get_metrics(self, metrics_pattern):
        params = {'query': metrics_pattern}
        return self.get("/metrics/find", None, params, 3)

    def get_datapoints(self, target_metrics_pattern, prior_days):
        params = {'target' : target_metrics_pattern,
                  'noNullPoints' : True,
                  'from' : '-%dd' % prior_days,
                  'until': 'now',
                  'format' : 'json',
                  'maxDataPoints': OMLGraphite.MAX_DATA_POINTS_PER_DAY*prior_days}
        return self.get("/render", None, params, 30)
