import functools

import os

import pymysql

from datasource import OMLJira, OMLInternalStatus, OMLCodeCov, OMLCapacity
from sandstormhelper import SandStormHelper


class IngestHandlers(object):
    """
    Handlers invoke data sources with the input parameters passed from the invoker and returns metric of interest.

    Handlers do not perform presentation work.

    Handlers maintain input/output at OML level (eg org structure)
    """
    @staticmethod
    def availability(rollup, bu, team, service):
        report_start_date, data = OMLInternalStatus().get_services_operate_within_an_availability_SLA(rollup, bu, team,
                                                                                                      service)
        return report_start_date, data

    @staticmethod
    def ri(rollup, bu, team, service):
        """
        Queries RI metrics
        :param rollup:  rollup (bu or team)
        :param bu: BU name
        :param team: team name
        :param service: service name (Unused as of 9/2017)
        :return: created, resolved
        """
        created, resolved = OMLJira().get_percent_RIs_resolved_in_timely_manner(rollup, bu, team)
        return created, resolved

    @staticmethod
    def bugs(rollup, bu, team, service):
        """
        Queries sev123 prod bugs metrics

        :param rollup:  rollup (bu or team)
        :param bu: BU name
        :param team: team name
        :param service: service name (Unused as of 9/2017)
        :return: created, resolved
        :return: created, resolved
        """
        created, resolved = OMLJira().get_percent_sev123_prod_bugs_resolved_in_timely_manner(rollup, bu, team)
        return created, resolved

    @staticmethod
    def alerts(rollup, bu, team, service):
        return OMLJira().get_incidents(rollup, bu)


class DBHelper(object):
    @staticmethod
    @functools.lru_cache()
    def get_connection():
        db_host = os.getenv('ALEMBIC_HOST', None)
        db_user = os.getenv('ALEMBIC_USER', None)
        db_password = os.getenv('ALEMBIC_PASSWORD', None)
        if db_host is None or db_user is None or db_password is None:
            db_host = SandStormHelper.get_secrets_str('qa-eng/oml/%s/dbhost' % os.environ['environment'])
            db_user = SandStormHelper.get_secrets_str('qa-eng/oml/%s/dbuser' % os.environ['environment'])
            db_password = SandStormHelper.get_secrets_str('qa-eng/oml/%s/dbpassword' % os.environ['environment'])

        return pymysql.connect(host=db_host, user=db_user, password=db_password, db='oml')

    @staticmethod
    def run_query(query, params=None):
        """
        Returns results in raw structure (tuple of tuples)

        :param query:
        :param params:
        :return:
        """
        db = DBHelper.get_connection()
        cur = db.cursor()
        if params is None:
            # print("INFO: Invoking query: %s" % query)
            cur.execute(query)
        else:
            # print("INFO: Invoking query: %s with params %s" % (query, params))
            cur.execute(query, params)
        db.commit()
        results = cur.fetchall()
        return results

    @staticmethod
    def run_query_flat_results(query, params=None):
        """
        Returns a flattened list of results. Use this for convenience if you are sure that the result can be flattened

        :param query:
        :param params:
        :return:
        """
        result = list(sum(DBHelper.run_query(query, params), ()))
        if result == [None]:
            return []
        else:
            return result

    @staticmethod
    @functools.lru_cache()
    def get_time_series_def_id(time_series_def_key):
        return DBHelper.run_query("SELECT ID FROM TIME_SERIES_DEF WHERE `KEY`='%s'" % time_series_def_key)[0][0]

    @staticmethod
    def most_recent_ingest_time(time_series_def_key):
        ts_def_id = DBHelper.get_time_series_def_id(time_series_def_key)
        return DBHelper.run_query("SELECT MAX(INGEST_BATCH) FROM TIME_SERIES_RAW WHERE TIME_SERIES_DEF_ID='%s'"
                                  % ts_def_id)[0][0]

    @staticmethod
    @functools.lru_cache()
    def get_rollup_id(rollup):
        return DBHelper.run_query("SELECT ID FROM ROLLUP WHERE `NAME`='%s'" % rollup)[0][0]


class QueryHandlers(object):
    """
    QueryHandlers has static methods for ready-only queries from the OML database

    """

    DOUBLE_TIME_SERIES_QUERY_BU_ROLLUP = """SELECT TSR1.INGEST_BATCH, OB.NAME, SUM(TSR1.VALUE), SUM(TSR2.VALUE)
FROM OML_BU OB
JOIN OML_TEAM OT ON OB.ID=OT.OML_BU_ID
JOIN TIME_SERIES_RAW TSR1 ON (OB.ID=TSR1.OML_BU_ID AND OT.ID=TSR1.OML_TEAM_ID)
JOIN TIME_SERIES_RAW TSR2 ON (OB.ID=TSR2.OML_BU_ID AND OT.ID=TSR2.OML_TEAM_ID)
WHERE TSR1.TIME_SERIES_DEF_ID=%s AND TSR1.INGEST_BATCH='%s' 
AND TSR2.TIME_SERIES_DEF_ID=%s AND TSR2.INGEST_BATCH='%s'
GROUP BY TSR1.INGEST_BATCH, OB.NAME"""

    # Use this query if metrics are unavailable at team level
    DOUBLE_TIME_SERIES_QUERY_BU_ROLLUP_BU_ONLY = """SELECT TSR1.INGEST_BATCH, OB.NAME, SUM(TSR1.VALUE), SUM(TSR2.VALUE)
FROM OML_BU OB
JOIN TIME_SERIES_RAW TSR1 ON OB.ID=TSR1.OML_BU_ID
JOIN TIME_SERIES_RAW TSR2 ON OB.ID=TSR2.OML_BU_ID
WHERE TSR1.TIME_SERIES_DEF_ID=%s AND TSR1.INGEST_BATCH='%s' 
AND TSR2.TIME_SERIES_DEF_ID=%s AND TSR2.INGEST_BATCH='%s'
GROUP BY TSR1.INGEST_BATCH, OB.NAME"""

    DOUBLE_TIME_SERIES_QUERY_TEAM_ROLLUP = """SELECT TSR1.INGEST_BATCH, OB.NAME, OT.NAME, SUM(TSR1.VALUE), SUM(TSR2.VALUE)
FROM OML_BU OB
JOIN OML_TEAM OT ON OB.ID=OT.OML_BU_ID
JOIN TIME_SERIES_RAW TSR1 ON (OB.ID=TSR1.OML_BU_ID AND OT.ID=TSR1.OML_TEAM_ID)
JOIN TIME_SERIES_RAW TSR2 ON (OB.ID=TSR2.OML_BU_ID AND OT.ID=TSR2.OML_TEAM_ID)
WHERE TSR1.TIME_SERIES_DEF_ID=%s AND TSR1.INGEST_BATCH='%s' 
AND TSR2.TIME_SERIES_DEF_ID=%s AND TSR2.INGEST_BATCH='%s'
GROUP BY TSR1.INGEST_BATCH, OB.NAME, OT.NAME"""

    DOUBLE_TIME_SERIES_QUERY_REPO_ROLLUP = """SELECT 
TSR1.INGEST_BATCH, OB.NAME, OT.NAME, GR.NAME, SUM(TSR1.VALUE), SUM(TSR2.VALUE)
FROM OML_BU OB
JOIN OML_TEAM OT ON OB.ID=OT.OML_BU_ID
JOIN GHE_REPO GR ON OT.ID=GR.OML_TEAM_ID
JOIN TIME_SERIES_RAW TSR1 ON (OB.ID=TSR1.OML_BU_ID AND OT.ID=TSR1.OML_TEAM_ID AND GR.ID=TSR1.GHE_REPO_ID)
JOIN TIME_SERIES_RAW TSR2 ON (OB.ID=TSR2.OML_BU_ID AND OT.ID=TSR2.OML_TEAM_ID AND GR.ID=TSR2.GHE_REPO_ID)
WHERE TSR1.TIME_SERIES_DEF_ID=%s AND TSR1.INGEST_BATCH='%s' 
AND TSR2.TIME_SERIES_DEF_ID=%s AND TSR2.INGEST_BATCH='%s'
GROUP BY TSR1.INGEST_BATCH, OB.NAME, OT.NAME, GR.NAME"""

    DOUBLE_TIME_SERIES_QUERY_CATALOG_SERVICE_ROLLUP = """SELECT 
TSR1.INGEST_BATCH, OB.NAME, OT.NAME, OS.NAME, SUM(TSR1.VALUE), SUM(TSR2.VALUE)
FROM OML_BU OB
JOIN OML_TEAM OT ON OB.ID=OT.OML_BU_ID
JOIN OML_SERVICE OS ON OT.ID=OS.OML_TEAM_ID
JOIN TIME_SERIES_RAW TSR1 ON (OB.ID=TSR1.OML_BU_ID AND OT.ID=TSR1.OML_TEAM_ID AND OS.ID=TSR1.OML_SERVICE_ID)
JOIN TIME_SERIES_RAW TSR2 ON (OB.ID=TSR2.OML_BU_ID AND OT.ID=TSR2.OML_TEAM_ID AND OS.ID=TSR2.OML_SERVICE_ID)
WHERE TSR1.TIME_SERIES_DEF_ID=%s AND TSR1.INGEST_BATCH='%s' 
AND TSR2.TIME_SERIES_DEF_ID=%s AND TSR2.INGEST_BATCH='%s'
GROUP BY TSR1.INGEST_BATCH, OB.NAME, OT.NAME, OS.NAME"""

    SINGLE_SERIES_QUERY_BOOLEAN_BU_ROLLUP = """SELECT TSR.INGEST_BATCH, OB.NAME, VALUE, COUNT(VALUE)
FROM OML_BU OB
JOIN OML_TEAM OT ON OB.ID=OT.OML_BU_ID
JOIN TIME_SERIES_RAW TSR ON (OB.ID=TSR.OML_BU_ID AND OT.ID=TSR.OML_TEAM_ID)
WHERE TSR.TIME_SERIES_DEF_ID=%s AND TSR.INGEST_BATCH='%s'
GROUP BY TSR.INGEST_BATCH, OB.NAME, VALUE
ORDER BY TSR.INGEST_BATCH, OB.NAME, VALUE"""

    SINGLE_SERIES_QUERY_BOOLEAN_TEAM_ROLLUP = """SELECT TSR.INGEST_BATCH, OB.NAME, OT.NAME, VALUE, COUNT(VALUE)
FROM OML_BU OB
JOIN OML_TEAM OT ON OB.ID=OT.OML_BU_ID
JOIN TIME_SERIES_RAW TSR ON (OB.ID=TSR.OML_BU_ID AND OT.ID=TSR.OML_TEAM_ID)
WHERE TSR.TIME_SERIES_DEF_ID=%s AND TSR.INGEST_BATCH='%s'
GROUP BY TSR.INGEST_BATCH, OB.NAME, OT.NAME, VALUE
ORDER BY TSR.INGEST_BATCH, OB.NAME, OT.NAME, VALUE"""

    @staticmethod
    def low_level_2_time_series_metrics(time_series_key_1, time_series_key_2, rollup, bu, team, service):
        time_series_id_1 = DBHelper.get_time_series_def_id(time_series_key_1)
        time_series_id_2 = DBHelper.get_time_series_def_id(time_series_key_2)
        mr_ingest_time = DBHelper.most_recent_ingest_time(time_series_key_2)

        if rollup == "bu":
            results = DBHelper.run_query(QueryHandlers.DOUBLE_TIME_SERIES_QUERY_BU_ROLLUP
                                    % (time_series_id_1, mr_ingest_time, time_series_id_2, mr_ingest_time))
        elif rollup == "team":
            results = DBHelper.run_query(QueryHandlers.DOUBLE_TIME_SERIES_QUERY_TEAM_ROLLUP
                                    % (time_series_id_1, mr_ingest_time, time_series_id_2, mr_ingest_time))
        elif rollup == "repo":
            results = DBHelper.run_query(QueryHandlers.DOUBLE_TIME_SERIES_QUERY_REPO_ROLLUP
                                         % (time_series_id_1, mr_ingest_time, time_series_id_2, mr_ingest_time))

        return results

    @staticmethod
    def availability(rollup, bu, team, service):
        ts_def_avail_id = DBHelper.get_time_series_def_id('AVAIL')
        ts_def_avail_sla_id = DBHelper.get_time_series_def_id('AVAIL-SLA')
        mr_ingest_time = DBHelper.most_recent_ingest_time('AVAIL')
        if rollup == "bu":
            avail_results = DBHelper.run_query(QueryHandlers.DOUBLE_TIME_SERIES_QUERY_CATALOG_SERVICE_ROLLUP
                                               % (ts_def_avail_id, mr_ingest_time, ts_def_avail_sla_id, mr_ingest_time))
        elif rollup == "team":
            avail_results = DBHelper.run_query(QueryHandlers.DOUBLE_TIME_SERIES_QUERY_CATALOG_SERVICE_ROLLUP
                                               % (ts_def_avail_id, mr_ingest_time, ts_def_avail_sla_id, mr_ingest_time))

        return avail_results

    @staticmethod
    def ri(rollup, bu, team, service):
        return QueryHandlers.low_level_2_time_series_metrics('RI-CLOSED', 'RI-ALL', rollup, bu, team, service)

    @staticmethod
    def bugs(rollup, bu, team, service):
        return QueryHandlers.low_level_2_time_series_metrics('S123CLOSED', 'S123ALL', rollup, bu, team, service)

    @staticmethod
    def codecov(rollup, bu, team, service):
        return QueryHandlers.low_level_2_time_series_metrics('CC-LOC-HIT', 'CC-LOC-ALL', "repo", bu, team, service)

    @staticmethod
    def alerts(rollup, bu, team, service):
        inc_all = DBHelper.get_time_series_def_id('INC-ALL-1K')
        inc_with_alerts = DBHelper.get_time_series_def_id('INC-ALRT-1K')
        mr_ingest_time = DBHelper.most_recent_ingest_time('INC-ALL-1K')

        if rollup == "bu":
            alerts_results = DBHelper.run_query(QueryHandlers.DOUBLE_TIME_SERIES_QUERY_BU_ROLLUP_BU_ONLY
                                    % (inc_with_alerts, mr_ingest_time, inc_all, mr_ingest_time))
        else:
            raise Exception("Incident Alerts not (yet) available at Team level")

        return alerts_results

    @staticmethod
    def configmgmt(rollup, bu, team, service):
        hosts_all = DBHelper.get_time_series_def_id('CFG-TOTAL')
        hosts_managed = DBHelper.get_time_series_def_id('CFG-MANAGED')
        mr_ingest_time = DBHelper.most_recent_ingest_time('CFG-TOTAL')

        if rollup == "bu":
            alerts_results = DBHelper.run_query(QueryHandlers.DOUBLE_TIME_SERIES_QUERY_BU_ROLLUP_BU_ONLY
                                    % (hosts_managed, mr_ingest_time, hosts_all, mr_ingest_time))
        else:
            raise Exception("Config Management currently only available at BU level")

        return alerts_results

    @staticmethod
    def binary_metrics(time_series_def_name, rollup, bu, team, service):
        tsr_def_id = DBHelper.get_time_series_def_id(time_series_def_name)
        mr_ingest_time = DBHelper.most_recent_ingest_time(time_series_def_name)
        binary_results = None

        if rollup == "bu":
            binary_results = DBHelper.run_query(QueryHandlers.SINGLE_SERIES_QUERY_BOOLEAN_BU_ROLLUP
                                    % (tsr_def_id, mr_ingest_time))
        elif rollup == "team":
            binary_results = DBHelper.run_query(QueryHandlers.SINGLE_SERIES_QUERY_BOOLEAN_TEAM_ROLLUP
                                    % (tsr_def_id, mr_ingest_time))

        return binary_results

    @staticmethod
    def unittest(rollup, bu, team, service):
        unittest_id = DBHelper.get_time_series_def_id('UNITTEST')
        mr_ingest_time = DBHelper.most_recent_ingest_time('UNITTEST')
        unittest_results = None

        if rollup == "bu":
            unittest_results = DBHelper.run_query(QueryHandlers.SINGLE_SERIES_QUERY_BOOLEAN_BU_ROLLUP
                                    % (unittest_id, mr_ingest_time))
        elif rollup == "team":
            unittest_results = DBHelper.run_query(QueryHandlers.SINGLE_SERIES_QUERY_BOOLEAN_TEAM_ROLLUP
                                    % (unittest_id, mr_ingest_time))

        return unittest_results

    @staticmethod
    def canaries(rollup, bu, team, service):
        return QueryHandlers.binary_metrics("CANARY", rollup, bu, team, service)

    @staticmethod
    def integtest(rollup, bu, team, service):
        return QueryHandlers.binary_metrics("INTEGTEST", rollup, bu, team, service)

    @staticmethod
    def autoenv(rollup, bu, team, service):
        return QueryHandlers.binary_metrics("AEC", rollup, bu, team, service)

    QUERY_2_TIME_SERIES_RATIO_BY_TEAM_AND_DATE = """SELECT TSR1.VALUE, TSR2.VALUE, TSR1.VALUE/TSR2.VALUE
FROM OML_TEAM OT 
JOIN TIME_SERIES_RAW TSR1 ON TSR1.OML_TEAM_ID=OT.ID
JOIN TIME_SERIES_RAW TSR2 ON (TSR2.OML_TEAM_ID=OT.ID AND TSR1.INGEST_BATCH=TSR2.INGEST_BATCH)
WHERE TSR1.TIME_SERIES_DEF_ID=%s
AND TSR2.TIME_SERIES_DEF_ID=%s
AND OT.ID=%s
AND TSR1.INGEST_BATCH > '%s'
ORDER BY TSR1.INGEST_BATCH DESC LIMIT 1"""
    @staticmethod
    def get_ri_closed_for_team(team_id, date_str):
        """
        Returns: All RIs are closed within SLA

        :param team_id: team id
        :param date_str: date string, in yyyy-mm-dd format
        :return:
        """
        tsr_def_ri_all = DBHelper.get_time_series_def_id('RI-ALL')
        tsr_def_ri_closed = DBHelper.get_time_series_def_id('RI-CLOSED')
        DBHelper.run_query(QueryHandlers.QUERY_2_TIME_SERIES_RATIO_BY_TEAM_AND_DATE
                           % (tsr_def_ri_closed, tsr_def_ri_all, team_id, date_str))

    @staticmethod
    def get_bugs_closed_for_team(team_id, date_str):
        """
        Returns: All RIs are closed within SLA

        :param team_id: team id
        :param date_str: date string, in yyyy-mm-dd format
        :return:
        """
        tsr_def_ri_all = DBHelper.get_time_series_def_id('RI-ALL')
        tsr_def_ri_closed = DBHelper.get_time_series_def_id('RI-CLOSED')
        DBHelper.run_query(QueryHandlers.QUERY_2_TIME_SERIES_RATIO_BY_TEAM_AND_DATE
                           % (tsr_def_ri_closed, tsr_def_ri_all, team_id, date_str))