from multiprocessing import Process, Pipe


def wrap_function_call_in_connection(*args):
    """
    This wraps function calls so that the output is sent to a Connection object.
    This is done as part of refactoring to use Pipe instead of Pool for multiprocessing, as the latter is disabled
    in post-Meltdown-patch AWS images.

    :param args: tuple (target, connection, args to target...)

    """
    target = args[0]
    connection = args[1]
    args_to_target = args[2:]
    if len(args_to_target) == 1:
        output = target(args_to_target[0])
    else:
        output = target(*args_to_target)
    connection.send(output)
    connection.close()


def chunks(l, n):
    """Yield successive n-sized chunks from l."""
    for i in range(0, len(l), n):
        yield l[i:i + n]


def run_parallel(target_function, list_of_argument_sets, max_concurrency):
    """
    Runs target function in parallel.

    See https://aws.amazon.com/blogs/compute/parallel-processing-in-python-with-aws-lambda/

    :param max_concurrency:
    :param target_function:
    :param list_of_argument_sets:
    :return:
    """
    outputs = []
    for arg_set_sublist in list(chunks(list_of_argument_sets, max_concurrency)):
        processes = []
        parent_connections = []
        for arg_set in arg_set_sublist:
            parent_conn, child_conn = Pipe()
            parent_connections.append(parent_conn)
            if type(arg_set) == tuple:
                args_to_wrapper = (target_function, child_conn) + arg_set
            else:
                args_to_wrapper = (target_function, child_conn) + (arg_set,)

            process = Process(target=wrap_function_call_in_connection, args=args_to_wrapper)
            processes.append(process)

        for process in processes:
            process.start()

        for parent_connection in parent_connections:
            output_from_parent_conn = parent_connection.recv()
            outputs.append(output_from_parent_conn)

        # make sure that all processes have finished
        for process in processes:
            process.join()

    return outputs

