import functools

from handlers import DBHelper


class OrgHelper(object):
    """
    Helper to navigate the orgs/teams between different underlying systems (jira, codecov, etc).

    The static methods are cached as these values are not expected to change much, especially during the lifetime of the
    OML query sessions.

    """

    @staticmethod
    @functools.lru_cache()
    def ref_BUs():
        ref_bu_set = set()
        for result in DBHelper.run_query("SELECT DISTINCT(NAME) FROM OML_BU"):
            ref_bu_set.add(result[0])

        return ref_bu_set

    @staticmethod
    @functools.lru_cache()
    def jira_BUs():
        jira_bu_set = set()
        results = DBHelper.run_query("SELECT DISTINCT(NAME) FROM JIRA_CATEGORY")
        for result in results:
            jira_bu_set.add(result[0])

        return jira_bu_set

    @staticmethod
    @functools.lru_cache()
    def jira_teams():
        jira_team_set = set()
        for result in DBHelper.run_query("SELECT DISTINCT(NAME) FROM JIRA_PROJECT"):
            jira_team_set.add(result[0])

        return jira_team_set

    @staticmethod
    @functools.lru_cache()
    def ref_teams():
        ref_team_set = set()
        for result in DBHelper.run_query("SELECT DISTINCT(NAME) FROM OML_TEAM"):
            ref_team_set.add(result[0])

        return ref_team_set

    @staticmethod
    @functools.lru_cache()
    def ref_teams_in_ref_bu(ref_bu):
        ref_team_set = set()
        for result in DBHelper.run_query("SELECT OT.NAME FROM OML_TEAM OT "
                                         "JOIN OML_BU OB ON OT.OML_BU_ID=OB.ID "
                                         "WHERE OB.NAME='%s'" % ref_bu):
            ref_team_set.add(result[0])

        return ref_team_set

    @staticmethod
    @functools.lru_cache()
    def ref_services_in_ref_team(ref_bu, ref_team):
        ref_services_set = set()
        for result in DBHelper.run_query("SELECT OS.NAME FROM OML_BU OB "
                                         "JOIN OML_TEAM OT ON OT.OML_BU_ID=OB.ID "
                                         "JOIN OML_SERVICE OS ON OS.OML_TEAM_ID=OT.ID "
                                         "WHERE OB.NAME='%s' AND OT.NAME='%s'" % (ref_bu, ref_team)):
            ref_services_set.add(result[0])

        return ref_services_set

    @staticmethod
    @functools.lru_cache()
    def jira_BU(ref_bu):
        jira_category = DBHelper.run_query("SELECT JC.NAME FROM JIRA_CATEGORY JC, OML_BU OB WHERE JC.OML_BU_ID=OB.ID AND OB.NAME='%s'" % ref_bu)
        if len(jira_category) > 0:
            return jira_category[0][0]

        return None

    @staticmethod
    @functools.lru_cache()
    def jira_team(ref_team):
        jira_project = DBHelper.run_query("SELECT JP.NAME FROM JIRA_PROJECT JP, OML_TEAM OT WHERE JP.OML_TEAM_ID=OT.ID AND OT.NAME='%s'" % ref_team)
        if len(jira_project) > 0:
            return jira_project[0][0]

        return None

    @staticmethod
    @functools.lru_cache()
    def jira_teams_in_ref_BU(ref_BU):
        teams = []
        for jira_proj in DBHelper.run_query("SELECT JP.NAME FROM OML_BU OB LEFT JOIN OML_TEAM OT ON OB.ID=OT.OML_BU_ID LEFT JOIN JIRA_PROJECT JP ON OT.ID=JP.OML_TEAM_ID WHERE OB.NAME='%s'" % ref_BU):
            teams.append(jira_proj[0])

        return teams

    @staticmethod
    @functools.lru_cache()
    def jira_to_ref(jira_team):
        results = DBHelper.run_query("SELECT OB.NAME, OT.NAME FROM JIRA_PROJECT JP LEFT JOIN OML_TEAM OT ON JP.OML_TEAM_ID=OT.ID LEFT JOIN OML_BU OB ON OT.OML_BU_ID=OB.ID WHERE JP.NAME='%s'" % jira_team)
        if len(results) > 0:
            return results[0]

        return None, None

    @staticmethod
    @functools.lru_cache()
    def jira_bu_to_ref_bu(jira_BU):
        results = DBHelper.run_query("SELECT OB.NAME FROM JIRA_CATEGORY JC LEFT JOIN OML_BU OB ON JC.OML_BU_ID=OB.ID WHERE JC.NAME='%s'" % jira_BU)
        if len(results) > 0:
            return results[0]

        return None


    @staticmethod
    @functools.lru_cache()
    def status_BU(ref_bu):
        results = DBHelper.run_query("SELECT SB.EXT_KEY FROM STATUS_BU SB, OML_BU OB WHERE SB.OML_BU_ID=OB.ID AND OB.NAME='%s'" % ref_bu)
        if len(results) > 0:
            return results[0][0]

        return None

    @staticmethod
    @functools.lru_cache()
    def status_team(ref_team):
        results = DBHelper.run_query("SELECT ST.NAME FROM STATUS_TEAM ST, OML_TEAM OT WHERE ST.OML_TEAM_ID=OT.ID AND OT.NAME='%s'" % ref_team)
        if len(results) > 0:
            return results[0][0]

        return None


    @staticmethod
    @functools.lru_cache()
    def status_team_to_ref_team(status_team):
        results = DBHelper.run_query("SELECT OT.NAME FROM STATUS_TEAM ST LEFT JOIN OML_TEAM OT ON ST.OML_TEAM_ID=OT.ID LEFT JOIN OML_BU OB ON OT.OML_BU_ID=OB.ID WHERE ST.NAME='%s'" % status_team)
        if len(results) > 0:
            return results[0][0]

        return None


    @staticmethod
    @functools.lru_cache()
    def status_teams_in_ref_BU(ref_BU):
        teams = []
        for status_team in DBHelper.run_query("SELECT ST.NAME FROM OML_BU OB "
                                              "LEFT JOIN OML_TEAM OT ON OB.ID=OT.OML_BU_ID "
                                              "LEFT JOIN STATUS_TEAM ST ON OT.ID=ST.OML_TEAM_ID "
                                              "WHERE OB.NAME='%s' AND ST.NAME IS NOT NULL" % ref_BU):
            teams.append(status_team[0])

        return teams


    @staticmethod
    @functools.lru_cache()
    def incident_BUs():
        incident_BU_set = set()
        results = DBHelper.run_query("SELECT DISTINCT(NAME) FROM INCIDENT_ORG")
        for result in results:
            incident_BU_set.add(result[0])

        return incident_BU_set

    @staticmethod
    @functools.lru_cache()
    def incident_to_ref_bu(incident_bu):
        results = DBHelper.run_query("SELECT OB.NAME FROM INCIDENT_ORG IO JOIN OML_BU OB ON IO.OML_BU_ID=OB.ID "
                                      "WHERE IO.NAME='%s'" % incident_bu)
        if len(results) > 0:
            return results[0][0]

        return None

    @staticmethod
    @functools.lru_cache()
    def ghe_repos_in_ref_team(ref_team):
        team_id = DBHelper.run_query_flat_results("SELECT OT.ID FROM OML_TEAM OT WHERE OT.NAME='%s'" % ref_team)[0]

        # get latest state of repos/teams structure
        all_repos = DBHelper.run_query("SELECT GR.NAME, GR.OML_TEAM_ID, GR.ID "
                                       "FROM GHE_REPO GR ORDER BY GR.NAME ASC, GR.ID DESC")
        repos = set()
        last_repo_name = ""
        for ghe_repo_name, oml_team_id, ghe_repo_id in all_repos:
            if ghe_repo_name == last_repo_name:
                continue  # SKIP if processing the same git repo name (which has a smaller ID, ie an older one)
            if oml_team_id == team_id:
                repos.add(ghe_repo_name)
            last_repo_name = ghe_repo_name

        return repos

    @staticmethod
    @functools.lru_cache()
    def status_to_ref_service(status_service):
        results = DBHelper.run_query("SELECT OS.NAME FROM STATUS_SERVICE SS "
                                     "JOIN OML_SERVICE OS "
                                     "ON SS.OML_SERVICE_ID=OS.ID WHERE SS.NAME='%s'" % status_service)
        if len(results) > 0:
            return results[0][0]

        return None

    @staticmethod
    @functools.lru_cache()
    def capacity_to_ref_bu(capacity_bu):
        results = DBHelper.run_query("SELECT OB.NAME FROM CAPACITY_BU CB "
                                     "JOIN OML_BU OB "
                                     "ON CB.OML_BU_ID=OB.ID WHERE CB.NAME='%s'" % capacity_bu)
        if len(results) > 0:
            return results[0][0]

        return None

    @staticmethod
    @functools.lru_cache()
    def inc_components_in_ref_team(ref_team):
        comps = set()
        results = DBHelper.run_query("SELECT IC.NAME FROM INCIDENT_COMP IC "
                                     "JOIN OML_TEAM OT ON IC.OML_TEAM_ID=OT.ID "
                                     "WHERE OT.NAME='%s'" % ref_team)
        if len(results) > 0:
            for entry in results:
                comps.add(entry[0])

        return comps