﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;
using System.Threading;
using System.IO;
using Newtonsoft.Json;

namespace ResolutionConsole
{
    class Program
    {

        #region Win32 API Imports

        [DllImport("User32.dll")]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern Boolean EnumDisplaySettings(
            [param: MarshalAs(UnmanagedType.LPTStr)]
            string lpszDeviceName,
            [param: MarshalAs(UnmanagedType.U4)]
            int iModeNum,
            [In, Out]
            ref DEVMODE lpDevMode);

        [DllImport("User32.dll")]
        [return: MarshalAs(UnmanagedType.I4)]
        public static extern int ChangeDisplaySettings(
            [In, Out]
        ref DEVMODE lpDevMode,
            [param: MarshalAs(UnmanagedType.U4)]
        uint dwflags);

        #endregion

        #region Structures

        [StructLayout(LayoutKind.Sequential,
        CharSet = CharSet.Ansi)]
        public struct DEVMODE
        {
            // You can define the following constant
            // but OUTSIDE the structure because you know
            // that size and layout of the structure
            // is very important
            // CCHDEVICENAME = 32 = 0x50
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
            public string dmDeviceName;
            // In addition you can define the last character array
            // as following:
            //[MarshalAs(UnmanagedType.ByValArray, SizeConst = 32)]
            //public Char[] dmDeviceName;

            // After the 32-bytes array
            [MarshalAs(UnmanagedType.U2)]
            public UInt16 dmSpecVersion;

            [MarshalAs(UnmanagedType.U2)]
            public UInt16 dmDriverVersion;

            [MarshalAs(UnmanagedType.U2)]
            public UInt16 dmSize;

            [MarshalAs(UnmanagedType.U2)]
            public UInt16 dmDriverExtra;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmFields;

            public POINTL dmPosition;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmDisplayOrientation;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmDisplayFixedOutput;

            [MarshalAs(UnmanagedType.I2)]
            public Int16 dmColor;

            [MarshalAs(UnmanagedType.I2)]
            public Int16 dmDuplex;

            [MarshalAs(UnmanagedType.I2)]
            public Int16 dmYResolution;

            [MarshalAs(UnmanagedType.I2)]
            public Int16 dmTTOption;

            [MarshalAs(UnmanagedType.I2)]
            public Int16 dmCollate;

            // CCHDEVICENAME = 32 = 0x50
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
            public string dmFormName;
            // Also can be defined as
            //[MarshalAs(UnmanagedType.ByValArray,
            //    SizeConst = 32, ArraySubType = UnmanagedType.U1)]
            //public Byte[] dmFormName;

            [MarshalAs(UnmanagedType.U2)]
            public UInt16 dmLogPixels;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmBitsPerPel;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmPelsWidth;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmPelsHeight;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmDisplayFlags;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmDisplayFrequency;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmICMMethod;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmICMIntent;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmMediaType;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmDitherType;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmReserved1;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmReserved2;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmPanningWidth;

            [MarshalAs(UnmanagedType.U4)]
            public UInt32 dmPanningHeight;
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct POINTL
        {
            [MarshalAs(UnmanagedType.I4)]
            public int x;
            [MarshalAs(UnmanagedType.I4)]
            public int y;
        }

        #endregion

        static void Main(string[] args)
        {
            IDictionary<string, string> arguments;
            arguments = parseArgs(args);

            string operation;
            if (arguments.TryGetValue("operation", out operation))
            {
                switch (operation)
                {
                    case "get":
                        GetCurrentSettings();
                        break;
                    case "set":
                        if (CallSetDisplay(arguments) == 0) // Call SetDisplay, if returned 0, exit success.
                        {
                            Environment.Exit(0);
                        }
                        else // Exit as failed
                        {
                            Environment.Exit(1);
                        }
                        break;
                    case "test":
                        TestAllSettings();
                        break;
                }
            }
        }

        // Gets the current resolution settings
        public static DEVMODE GetCurrentSettings()
        {
            DEVMODE mode = new DEVMODE();
            mode.dmSize = (ushort)Marshal.SizeOf(mode);
            EnumDisplaySettings(null, -1, ref mode);
            WriteLog("Current Resolution: " + mode.dmPelsWidth + " x " + mode.dmPelsHeight, "INFO");
            return mode;
        }

        // This method will test all resolution settings and record their result
        // It's good for determining the supported resolution of a graphics driver
        public static void TestAllSettings()
        {
            DEVMODE mode = new DEVMODE();
            mode.dmSize = (ushort)Marshal.SizeOf(mode);

            int modeIndex = 0; // Start at 0
            // Loop through all supported Display Settings - when it returns false, it's done.
            while (EnumDisplaySettings(null, modeIndex, ref mode) == true)
            {
                WriteLog("Mode Index: " + modeIndex, "INFO");
                int result = SetDisplaySettings(mode.dmPelsWidth, mode.dmPelsHeight); // Set the mode's width and height
                WriteLog((modeIndex.ToString() + ": " + mode.dmPelsWidth.ToString() + " x " + mode.dmPelsHeight.ToString() + " Result: " + result.ToString()), "INFO");
                Thread.Sleep(3000); // Sleep for 3 seconds, to allow the resolution time to set
                modeIndex++; // Increment
            }
            WriteLog("Complete testing data", "INFO");
        }

        // Sets the display settings to a set resolution
        public static int SetDisplaySettings(uint width, uint height)
        {
            DEVMODE originalMode = GetCurrentSettings();
            DEVMODE newMode = originalMode; // Make a copy that can be modified

            // Update the settings
            newMode.dmPelsWidth = (uint)width;
            newMode.dmPelsHeight = (uint)height;

            WriteLog(("Preparing to set resolution, width: " + width.ToString() + " height: " + height.ToString()), "INFO");
            int result = ChangeDisplaySettings(ref newMode, 0); // Make the update, capture the result
            WriteLog(("Result of the above operation: " + result.ToString()), "INFO");
            return result;
        }

        // Calls the SetDisplay method, converting the arguments to uint
        public static int CallSetDisplay(IDictionary<string, string> arguments)
        {
            string width;
            string height;
            arguments.TryGetValue("width", out width);
            arguments.TryGetValue("height", out height);

            uint width_num = Convert.ToUInt32(width);
            uint height_num = Convert.ToUInt32(height);
            
            int result = SetDisplaySettings(width_num, height_num);
            return result;
        }

        // Parses the arguments, getting the operation, width, and height
        // Expected Example: set -width 800 -height 600
        public static IDictionary<string, string> parseArgs(string[] args)
        {
            string[] supported_args = new string[] { "get", "set", "test" };
            string operation = ""; // Used to store the operation type

            if (args.Length > 0)
            {
                operation = args[0]; // Assume the operation is the first argument.
            }
            else
            {
                WriteLog("Must provide an argument. Valid options: get, set, test", "ERROR");
                Environment.Exit(1);
            }

            if (!supported_args.Contains(operation)) // INVERSE - If supported args doesn't contain operation
            {
                WriteLog("Did not recognize the first argument. Must provide get, set, or test", "ERROR");
                Environment.Exit(1);
            }

            // Create a dictionary structure to store the arguments
            IDictionary<string, string> dict = new Dictionary<string, string>();
            dict.Add("operation", operation);

            for (int i = 0; i < args.Length; i++) // Loop through all arguments
            {
                string arg = args[i];
                if (arg[0] == '-') // If the first character is a -, it's an argument name
                {
                    string type = arg.Substring(1); // Grab everything after dash (-)

                    // Add the type to the dictionary, and the value as the next index
                    // TODO: Better validation, i.e., handle if user passed -width -height with no numbers
                    dict.Add(type, args[i + 1]);
                }
            }

            return dict;
        }

        private static void WriteLog(string message, string level)
        {
            var log = new Dictionary<string, Object>();
            DateTime timestamp = DateTime.Now.ToUniversalTime();

            log.Add("@timestamp", timestamp);
            log.Add("level", level);
            log.Add("message", message);

            string jsonLog = JsonConvert.SerializeObject(log);
            Console.WriteLine(jsonLog);
        }
    }
}
