import Config from '../config'
import FileSystem from '../lib/file-system'
import Logger from '../lib/logger'
import { handler } from './install'

exports.command = 'init'
exports.desc = 'Initializes a repo for Smoca development by creating a \'smoca\' directory.'
exports.builder = {
  d: {
    default: undefined,
    alias: 'dir',
    type: 'string'
  },
  v: {
    default: undefined,
    alias: 'ver',
    type: 'string'
  }
}
exports.handler = function (argv) {
  Config.setSmocaDir(argv.dir)
  Config.setSmocaVersion(argv.ver)

  // Handle the Smoca directory.
  if (FileSystem.doesDirExist(Config.smocaDirPath)) {
    Logger.warn(`Will not init. Smoca directory already exists:\n  ${Config.smocaDirPath}\n`)
    return
  } else {
    FileSystem.makeDir(Config.smocaDirPath)
    Logger.progress(`Created: ${Config.smocaDirPath}`)
  }

  // Create the .gitignore file if necessary.
  FileSystem.writeFileIfNotExist(Config.smocaDirPath + '.gitignore', 'tmp\n*~\n.DS_Store\n.idea\n')

  // Create the Ruby Version file.
  FileSystem.writeFileIfNotExist(Config.smocaDirPath + '.ruby-version', Config.rubyVersion)

  // Create the Ruby Gemfile file.
  let gemFile = "source 'http://rubygems.org'\n"
  gemFile += "source 'https://gem.internal.justin.tv' do\n"
  gemFile += "gem 'smoca'"
  // Set the Gemfile to use a specific version of Smoca if one was defined.
  if (Config.smocaVersion !== undefined) {
    gemFile += `, '${Config.smocaVersion}'`
    Logger.warn(`Using Smoca Gem v${Config.smocaVersion}`)
  }
  gemFile += '\nend'
  FileSystem.writeFileIfNotExist(Config.smocaDirPath + 'Gemfile', gemFile)

  // Create the Ruby Spec directory.
  const specDirPath = Config.smocaDirPath + 'spec'
  if (!FileSystem.doesDirExist(specDirPath)) {
    FileSystem.makeDir(specDirPath)
    Logger.progress(`Created: ${specDirPath}`)
  }

  // Create the spec_helper.rb
  let specHelper = "require 'smoca'\n\n"
  specHelper += "Smoca.app_host = 'https://www.twitch.tv'\n"
  FileSystem.writeFileIfNotExist(specDirPath + '/spec_helper.rb', specHelper)

  // Create Driver Config File
  const configPath = Config.smocaDirPath + 'configuration'
  if (!FileSystem.doesDirExist(configPath)) {
    FileSystem.makeDir(configPath)
    Logger.progress(`Created: ${configPath}`)
  }
  let driverConfig = `
GRID: false # [true, false]
GRID_HOST:  # [internal, internal_dev, saucelabs]

OS_PLATFORM:  # [Linux, Windows, macOS]
OS_VERSION:
BROWSER_NAME:  # [firefox, chrome]
BROWSER_VERSION:

SCREEN_RESOLUTION:

SAUCELABS_USERNAME: # DO NOT COMMIT CREDENTIALS
SAUCELABS_ACCESS_KEY: # DO NOT COMMIT CREDENTIALS
  `;
  FileSystem.writeFileIfNotExist(configPath + '/driver.yaml', driverConfig)

  // Now run `smoca install`
  handler(argv)
}
