import Config from '../config'
import FileSystem from '../lib/file-system'
import CmdLine from '../lib/cmd-line'
import Logger from '../lib/logger'
import { execSync } from 'child_process'
import colors from 'colors'

exports.command = 'install [dir]'
exports.desc = 'Installs Smoca and necessary Ruby dependencies based on the Smoca dir'
exports.builder = {}
exports.handler = function (argv) {
  // Alter the directory we'll use for installing Smoca if one was defined.
  Config.setSmocaDir(argv.dir)

  // Handle the Smoca directory.
  if (!FileSystem.doesDirExist(Config.smocaDirPath)) {
    Logger.warn(`Smoca dir not found:\n  ${Config.smocaDirPath}\nUse 'init' to create one.`)
    return
  }

  // Make sure rbenv is installed.
  if (!CmdLine.hasCmd('rbenv')) {
    if (!CmdLine.askUser('rbenv is not installed.  Would you like to install it?')) {
      Logger.error('rbenv not found.  Please install before continuing.', true)
      return
    }

    // Need homebrew to install Ruby.
    if (!CmdLine.hasCmd('brew')) {
      Logger.error('Homebrew not installed. Please install', true)
      return
    }
    CmdLine.runCmdAndExitOnError('brew update')
    CmdLine.runCmdAndExitOnError('brew install rbenv')
    CmdLine.runCmdAndExitOnError('brew install ruby-build')
    CmdLine.runCmdAndExitOnError("echo 'eval \"$(rbenv init -)\"' >> ~/.bash_profile")
    CmdLine.runCmdAndExitOnError('source ~/.bash_profile')
  }

  // Check the version of Ruby.
  let rubyVersionStr = ''
  try {
    rubyVersionStr = execSync('rbenv version', { cwd: Config.smocaDirPath })
    rubyVersionStr = rubyVersionStr.toString()
  } catch (err) {}

  if (!rubyVersionStr.startsWith(Config.rubyVersion)) {
    if (!CmdLine.askUser(`ruby version ${Config.rubyVersion} is not installed.  Would you like to install?`)) {
      Logger.error(`ruby version ${Config.rubyVersion} is not installed.`, true)
      return
    }
    CmdLine.runCmdAndExitOnError(`rbenv install ${Config.rubyVersion} -v`, Config.smocaDirPath)
    CmdLine.runCmdAndExitOnError(`rbenv rehash`, Config.smocaDirPath)
    CmdLine.runCmdAndExitOnError('gem update --system', Config.smocaDirPath)
    CmdLine.runCmdAndExitOnError('gem install bundler -v 1.11.2', Config.smocaDirPath)
    CmdLine.runCmdAndExitOnError('rbenv rehash', Config.smocaDirPath)
  }

  CmdLine.runCmdAndExitOnError('bundle install', Config.smocaDirPath)

  let successMsg = '\n'
  successMsg += '--------------------------------------------------\n'
  successMsg += ' Successfully installed Smoca.  Happy testing! :)\n'
  successMsg += '--------------------------------------------------'
  console.log(colors.green(successMsg))
}
