import { execSync } from 'child_process'
import colors from 'colors'

export default class CmdLine {
  /**
   * Checks to see if a given command is pathed on our command-line.
   * @param  {string}  cmdName Name of the command we want to look for.
   * @return {boolean}         Indicates whether or not the command was found.
   */
  static hasCmd (cmdName) {
    let cmdPath = ''
    try {
      cmdPath = execSync(`which ${cmdName}`)
      cmdPath = cmdPath.toString()
    } catch (err) {
      cmdPath = ''
    }
    if (cmdPath === '') {
      return false
    }
    return true
  }

  /**
   * Asks the user a yes no question and returns a boolean indicating what they answered.  If a user doesn't enter
   * anything then the default answer will be used.  if there is no default answer, then the user willb e prompted to
   * select a valid response.
   * @param  {string} question            Question we want to ask the user.
   * @param  {string} [defaultAnswer='y'] Default answer if user hits enter without answering can be 'y', 'n, or ''.
   * @return {boolean}                    Boolean indicating what the user selected: 'y' = true, 'n' = false
   */
  static askUser (question, defaultAnswer = 'y') {
    let prompt = require('prompt-sync')()
    let options = '[y/n]'
    defaultAnswer = defaultAnswer.toLowerCase()
    if (defaultAnswer === 'y') {
      options = '[Y/n]'
    } else if (defaultAnswer === 'n') {
      options = '[y/N]'
    }

    let answer
    try {
      answer = prompt(colors.bold(question + ' ') + colors.grey(options) + ' ')
    } catch(err) {
      // If not in an interactive shell, this will fail. Auto return whatever default
      return (defaultAnswer == 'y')
    }

    if (answer === '') {
      answer = defaultAnswer
    } else {
      answer = answer.toLowerCase()
    }
    if (answer === 'yes' || answer === 'y') {
      return true
    } else if (answer === 'no' || answer === 'n') {
      return false
    }
    console.log(colors.bold.red("Invalid answer. Please enter 'yes' or 'no'"))
    return CmdLine.askUser(question)
  }

  /**
   * Runs a command and shows its output.  Returns the exit status of the command, and an error obj (if there is one).
   * @param  {string} cmdStr  Command with want to run (with its args).
   * @param  {string} cwdPath If set, then runs our cmd in this path, otherwise cmd will run in current working dir.
   * @return {Error}          Error obj from running the command (if there was one).
   */
  static runCmd (cmdStr, cwdPath) {
    try {
      console.log(colors.magenta('---> ' + cmdStr))
      let options = { stdio:[ 0, 1, 2 ] }
      if (cwdPath !== undefined) {
        options.cwd = cwdPath
      }
      execSync(cmdStr, options)
    } catch (e) {
      return e
    }
    return undefined
  }

  /**
   * Runs a command and exits the application if an error was encountered.
   * @param  {string} cmdStr  Command with want to run (with its args).
   * @param  {string} cwdPath If set, then runs our cmd in this path, otherwise cmd will run in current working dir.
   */
  static runCmdAndExitOnError (cmdStr, cwdPath) {
    let error = CmdLine.runCmd(cmdStr, cwdPath)
    if (error !== undefined) {
      console.log(colors.red(`Failed to run: ${cmdStr}`))
      process.exit(1)
    }
  }
}
