import Logger from './logger'
import fs from 'file-system'

export default class FileSystem {
  /**
   * Provided a path, checks to see if the path is a directory that exist.
   * @param  {string} path Path to directory we want to check.
   * @return {boolean}     Indicates whether or not the directory exist.
   */
  static doesDirExist (path) {
    try {
      return fs.statSync(path).isDirectory()
    } catch (e) {
      if (e.code === 'ENOENT') {
        return false
      } else {
        throw e
      }
    }
  }

  /**
   * Provided a path, checks to see if the path is a file that exist.
   * @param  {string} path Path to a file we want to check.
   * @return {boolean}     Indicates whether or not the directory exist.
   */
  static doesFileExist (aPath) {
    try {
      return fs.statSync(aPath).isFile()
    } catch (e) {
      if (e.code === 'ENOENT') {
        return false
      } else {
        throw e
      }
    }
  }

  /**
   * Writes data to a file, but only if the file doesn't already exist.
   * @param  {string} filePath  File we want to write to.
   * @param  {string} writeData Data we want to write to the file.
   */
  static writeFileIfNotExist (filePath, writeData, showPath = true) {
    if (!FileSystem.doesFileExist(filePath)) {
      fs.writeFileSync(filePath, writeData)
    }
    if (showPath) {
      Logger.progress(`Created: ${filePath}`)
    }
  }

  /**
   * Creates a directory.
   * @param  {string} dirPath Path of the directory we want to create.
   */
  static makeDir (dirPath) {
    fs.mkdirSync(dirPath)
  }
}
