require 'yaml'
require_relative 'properties'

module Smoca
  class Configuration
    class << self
      attr_accessor :configuration_file_path

      # Initializes a configuration
      def initialize
        @configuration_file_path ||= './configuration/driver.yaml'
        parse_config(load_config_file)
      end

      # Parses through configuration values. Environment Variables take precedence over config file
      # @param config [Hash] The configuration file data
      private def parse_config(config)
        env_vars = ENV.to_hash

        config.each do |k, v|
          value = v
          if env_vars.key?(k)
            puts "[INFO] Overriding #{k}=#{value} with #{k}=#{env_vars[k]}"
            value = env_vars[k]
          end
          value = standardize_value(value)
          send_to_properties(k, value)
        end
      end

      # Sends a property to the Properties class
      # @param name [String] The name of the property to set
      # @param value [String] The value of the property to set
      private def send_to_properties(name, value)
        begin
          Smoca::Properties.send("#{name.downcase}=", value)
        rescue NoMethodError => e
          puts "[WARN] Attempted to register '#{name}' with Smoca Properties."\
            " Received #{e.class}: #{e.message}"
        end
      end

      # Standardize configuration values. Converts strings of 'true' and 'false' to booleans
      # @param value [String] The value to convert
      # @return [Boolean] if value was a string matching true or false
      # @return [String] if the value was not a recognized boolean convertable. Will be whatever the user passed in
      private def standardize_value(value)
        return value unless value.is_a?(String)
        case value.downcase
          when 'true'
            return true
          when 'false'
            return false
          else
            return value
        end
      end

      # @return [Hash] Data from the configuration file
      private def load_config_file
        if File.exist?(@configuration_file_path)
          return YAML.load_file(@configuration_file_path)
        else
          puts "[WARN] Unable to find configuration file at #{@configuration_file_path}."\
            " Loading Smoca with defaults."
          return {}
        end
      end

    end
  end
end
