require_relative 'base'
require_relative '../configuration/properties'

module Smoca
  class Driver
    class Remote < Base

      class << self
        DEFAULT_OPTIONS = {
          remote_type: :internal,
          # Saucelabs Credentials
          access_id: nil,
          access_key: nil
        }.freeze

        # @param capabilities [Smoca::Driver::RemoteCapabilities] The browser capabilities
        # @param opts [Hash] The options to initialize the driver with
        # @option opts [Symbol] :remote_type (:internal) The remote connection source
        #   Options: [:internal, :saucelabs, :internal_dev]
        # @option opts [Symbol] :access_id (nil) The Sauce Labs Access ID (username)
        # @option opts [Symbol] :access_key (nil) The Sauce Labs Access Key
        def initialize(capabilities, opts = {})
          @options = DEFAULT_OPTIONS.merge(opts)
          @capabilities = convert_caps(capabilities.caps)

          @remote_type = @options[:remote_type]
          @remote_type = @remote_type.to_sym unless @remote_type.nil?

          register_driver(driver_opts)
        end

        def convert_caps(caps)
          return caps unless Smoca::Properties.grid_host == 'saucelabs'

          if caps[:os_version]
            caps[:platform] = "#{caps[:platform]} #{caps[:os_version]}"
          end

          return caps
        end

        # @return [Hash] The formatted driver options for Capybara
        private def driver_opts
          client = Selenium::WebDriver::Remote::Http::Default.new
          client.read_timeout = 120 # Seconds. Sets for the Selenium Remote Driver.

          return {
            browser:              :remote,
            url:                  grid_url,
            http_client:          client,
            desired_capabilities: @capabilities
          }
        end

        # @return [String] The URL to connect to Grid
        private def grid_url
          urls = { internal: 'http://grid.us-west2.justin.tv:80/wd/hub',
                   internal_dev: 'http://grid-dev.us-west2.justin.tv:80/wd/hub',
                   saucelabs: saucelabs_url}

          # Fail if trying to retrieve a SauceLabs URL without access_id or access_key
          if sauce_requested? && !sauce_keys_provided?
            raise 'Must provide both access_id and access_key when initializing remote driver'
          end

          if urls.key?(@remote_type)
            return urls[@remote_type]
          else
            raise "Unexpected remote_type: #{@remote_type}. Valid entries: #{urls.keys}"
          end
        end

        # Method to error check if user did not provide access id or key
        # @return [String] Sauce Labs URL Endpoint
        private def saucelabs_url
          return "http://#{@options[:access_id]}:#{@options[:access_key]}@ondemand.saucelabs.com:80/wd/hub"
        end

        # @return [Boolean] If the user requested a Sauce Labs type
        private def sauce_requested?
          return @remote_type == :saucelabs
        end

        # @return [Boolean] If the user provided all Sauce Labs Keys
        private def sauce_keys_provided?
          return !(@options[:access_id].nil? && @options[:access_key].nil?)
        end
      end

    end
  end
end
