require 'dropbox_sdk'

module Smoca
  class DropboxConfig
    class << self
      # Lazy Evaluation. Once called, will initiate a connection with Dropbox
      # @return [DropboxClient] The Dropbox Client object to interact with.
      #   See: http://www.rubydoc.info/gems/dropbox-sdk/1.6.5/DropboxClient
      def client
        return nil unless token_provided?
        return @client ||= DropboxClient.new(token)
      end

      # @return [String] The dropbox token as provided by the user via Environment Variable DROPBOX_TOKEN
      def token
        return ENV['DROPBOX_TOKEN']
      end

      # @return [Boolean] Whether or not a Dropbox token was provided
      def token_provided?
        return !!token
      end

      # @param filepath [String] The path to the file to upload
      # @return [Hash] Hash of data returned by Dropbox. Recommend pulling :path
      def upload(filepath)
        return nil unless token_provided? && File.exists?(filepath)
        file = open(filepath)
        begin
          response = client.put_file(filepath, file)
        rescue Exception => e
          raise "When uploading file to Dropbox, received: [#{e.class}] #{e.message}. Ensure Dropbox token is correct."
        end

        return response
      end

      # @return [String] The Dropbox URL to the File
      def share_url(dropbox_filepath)
        return nil unless token_provided?
        return client.shares(dropbox_filepath)['url']
      end

      # Both uploads to dropbox and makes a public link
      # @param filepath [String] The path to the file on dropbox.
      #   By default it's the same a the relative path locally
      # @return [String] The Dropbox URL to the File
      def upload_and_share(filepath)
        dropbox_path = upload(filepath)['path']
        return share_url(dropbox_path)
      end
    end
  end
end
