require_relative '../dropbox_config'

class ScreenshotFormatter
  RSpec::Core::Formatters.register self, :example_failed, :dump_summary

  def initialize(notification)
  end

  # Outputs the image url
  # @param notification [RSpec::Core::Notifications::FailedExampleNotification]
  def example_failed(notification)
    screenshot_path = local_image_path(notification.example)
    return nil if screenshot_path.nil? # No screenshot, return

    if Smoca::DropboxConfig.token_provided? # Use Dropbox URL
      dropbox_url = Smoca::DropboxConfig.upload_and_share(screenshot_path)
      notification.example.metadata[:screenshot][:dropbox_image_url] = dropbox_url
    end

    img_path = preferred_image_path(notification.example)
    puts "\tScreenshot: #{img_path}" unless img_path.nil?
  end

  # Outputs each screenshot url next to the scenario description as a separate section.
  # Should be considered temporary until we add the screenshot directly with the failure dump summary
  # @param notification [RSpec::Core::Notifications::SummaryNotification]
  def dump_summary(notification)
    puts "\n" + section_break + "\nScreenshots:"
    notification.failed_examples.each do |ex|
      img_path = preferred_image_path(ex)
      puts "\t#{ex.full_description}: #{img_path}" unless img_path.nil?
    end
    puts section_break
  end

  # Returns the preferred path to an image.
  #   Preference 1: Dropbox
  #   Preference 2: Local
  # @param example [RSpec::Core::Example] The example to lookup the image
  # @return [String] if Dropbox is enabled, the dropbox url
  # @return [String] if Dropbox is disabled, the local file
  private def preferred_image_path(example)
    if dropbox_path = dropbox_image_path(example)
      return dropbox_path
    elsif local_path = local_image_path(example)
      return local_path
    else
      return nil
    end
  end

  # @param example [RSpec::Core::Example] The example to lookup the image
  # @return [String] if there is a local image path
  # @return [NilClass] if there is no image available
  private def local_image_path(example)
    return example.metadata.dig(:screenshot, :image)
  end

  # @param example [RSpec::Core::Example] The example to lookup the image
  # @return [String] if there is a local image path
  # @return [NilClass] if there is no image available
  private def dropbox_image_path(example)
    example.metadata.dig(:screenshot, :dropbox_image_url)
  end

  # @return [String] String with hyphens to signify a break in section
  # @param length [Int] The amount of hypthens to print
  private def section_break(length=35)
    return "#{'-' * length}"
  end
end
