# Utilities for dealing with Screenshot / HTML Storage
class ScreenshotUtils
  class << self

    # Searches metadata for failure screenshot
    # @param example [RSpec::Core::Example]
    # @return [String] The path to the failure screenshot
    def locate_failure_screenshot(example)
      return example.metadata.dig(:screenshot, :image) # Grabs the path to the image saved from the metadata
    end

    # @param example [RSpec::Core::Example]
    # @return [String] The path to the failure HTML
    def locate_failure_html(example)
      return example.metadata.dig(:screenshot, :html) # Grabs the path to the image saved from the metadata
    end

    # File path for where screenshots are stored
    # @return [String] The location where to store files
    def failure_location_path
      date = Time.now.strftime('%m_%-d_%Y') # mm_d_yyyy
      return "resources/output/failures/#{date}"
    end

    # @return [Boolean] Whether screenshotting is enabled
    def screenshots_enabled?
      return ENV['SCREENSHOTS'] == 'true'
    end

    # Converts a Raw S3 URL to the Smoca Screenshot Proxy
    # Details on Smoca Screenshot Proxy: https://git-aws.internal.justin.tv/qe/smoca_screenshots
    # @param original_url [String] The screenshot url, stored in S3
    # @return [String] if successful, the Smoca Screenshot URL
    # @return [NilClass] if an error occurred while converting the screenshot
    def convert_s3_url(original_url)
      host_to_remove = 'https://smoca-screenshots.s3.us-west-2.amazonaws.com'
      host_to_add    = 'http://qe-smoca-screenshots.us-west2.justin.tv'
      begin
        return original_url.gsub!(host_to_remove, host_to_add)
      rescue Exception => e
        puts "[WARN] When converting the url, encountered #{e.class} #{e.message}"
        return nil
      end
    end

  end
end
