require_relative 'screenshot_utils'
require_relative '../configs/s3_config'

# Miscellaneous Utilities class
class TwitchUtils

  # Evaluate a String as a Boolean
  def self.bool_value_of(str)
    return str == 'true'
  end
  # Determine whether or not a String is empty or nil
  def self.nil_or_empty?(str)
    return (str.to_s.empty?)
  end

  # Saves the data to a file
  # @return String Filepath of where the HAR was saved
  def self.save_file(data, directory_path, filename)
    FileUtils.mkdir_p(directory_path) unless Dir.exists?(directory_path) # Create the date folder unless it already exists
    filepath = "#{directory_path}/#{filename}"
    File.open(filepath, 'w') { |file| file.write(data) } # Create and write to the file.
    return filepath
  end

  # @param timeout Int Seconds until event listener should timeout
  def self.execute_event_listener_script(js, timeout)
    begin
      # Set script to timeout after waiting for a few seconds
      Capybara.page.driver.browser.manage.timeouts.script_timeout = timeout
      response = Capybara.page.driver.browser.execute_async_script(js)
      return response
    rescue Selenium::WebDriver::Error::TimeOutError
      puts '[WARNING]: Rescuing Selenium::WebDriver::Error::TimeOutError'
      return false
    rescue Selenium::WebDriver::Error::ScriptTimeoutError
      return false
    ensure
      # Set script back to original timeout. Need to refactor to non-hardcoded number.
      Capybara.page.driver.browser.manage.timeouts.script_timeout = 45
    end
  end

  # Remove the v from player_vod data.
  # @return [String] player_vod id without the v
  def self.vod_id(vod)
    regex = /[v]?(\d*)/
    return vod.match(regex)[1]
  end

  # Specifically for use with BrowserStack.
  # BrowserStack has a set IDLE TIMEOUT of 90 seconds.
  # This can not be altered on our end, so the test will need
  # to periodically send a request to BrowserStack in order to
  # prevent IDLE TIMEOUT from happening.
  #
  # Method will sleep the 'poll_wait' amount of time
  # in seconds up to the 'max_wait' of time in seconds.
  # After every 'poll_wait' 'current_url' is requested to keep
  # BrowserStack active.
  # @param poll_wait [Integer] Time to sleep in seconds.
  # @param max_wait [Integer] Max amount of time to sleep in seconds.
  def self.fluent_wait(poll_wait, max_wait)
    time = 0
    while time < max_wait
      sleep(poll_wait)
      Capybara.page.current_url
      time = time + poll_wait
    end
  end

  # Temp logging method - SMOC-330 BMP Connection Issues
  def self.debug_logging(response, event)
    filename = Time.now.strftime('%H_%M_%S') + "_#{event}"
    directory_path = ScreenshotUtils.failure_location_path
    filepath = TwitchUtils.save_file(response, directory_path, filename)
    url = S3Config.upload_file(filepath)
    puts "RESPONSE LOG: #{url}"
  end

  def self.log_errors(response, event, exception)
    puts "Encountered error #{exception.class}."
    TwitchUtils.debug_logging(response, event)
    raise exception
  end

  # Fetch the selected text from select dropdown
  # @param select_element [Capybara::Element] element locator for slect dropdown
  # @return [Capybara::Element] text of option selected in dropdown
  def self.selected_text(select_element)
    value = select_element.value
    options = select_element.all('option')
    options.each do |opt|
      return opt.text if opt.value.strip == value.strip
    end
  end

  # Method to return a formatted thread number
  # @return [String] The thread number for formatting purposes
  def self.thread_number
    if TwitchUtils.nil_or_empty?(ENV['TEST_ENV_NUMBER'])
      return '1' # The first process is blank, but give this a number for formatting.
    else
      return ENV['TEST_ENV_NUMBER']
    end
  end
end
