#!/usr/bin/env bash

# This Script is Responsible for:
# 1) Starting Sauce Connect / Browser Stack Local (if desired via LOCAL_TESTING==true)
# 2) Running our Test Suite
# 3) Closing Sauce Connect, if it was started

# We currently support both Sauce Connect and Browser Stack Local.


################################################################
#  Start Sauce Connect or Browser Stack Local (if applicable)  #
################################################################

if [[ $LOCAL_TESTING == true ]]; then
  # Assign a Unique ID for BrowserStack Local Testing / Sauce Connect to use
  # Eventually we'll probably want to change the variable to TUNNEL_ID for clarity, but wanted to stay in scope.
  BROWSERSTACK_LOCAL_ID=$(cat /dev/urandom | env LC_CTYPE=C tr -dc 'a-zA-Z0-9' | fold -w 32 | head -n 1)
  export BROWSERSTACK_LOCAL_ID=$BROWSERSTACK_LOCAL_ID
  export SAUCE_TUNNEL_IDENTIFIER=$BROWSERSTACK_LOCAL_ID

  # If BMP is true, open and assign a port
  if [[ $BMP == true ]]; then
    BMP_PORT="$(curl -X POST http://bmp.us-west2.justin.tv:9090/proxy | jq-linux64 -r '.port')"
    export BMP_PORT=$BMP_PORT
    echo "[INFO] Opened BrowserMob Proxy Port: ${BMP_PORT}"
  fi

  # Start Respective Binary, either Sauce Connect or BrowserStack Local
  if [[ $GRID_HOST == 'saucelabs' ]]; then
    echo "[INFO] Starting Sauce Connect"

    # Used for Sauce Connect to indicate it's ready.
    SAUCE_READY_FILE_PATH="/build/sauce_is_ready"

    # Building an options string to pass to Sauce Connect
    # Added --no-ssl-bump-domains flag to support websockets, which our chat uses.
    # "Sauce Connect by default uses SSL bumping, replacing SSL certificates with its own trusted cert. This can cause issues with websocket connections"
    options="-u $SAUCELABS_USERNAME \
      -k $SAUCELABS_ACCESS_KEY \
      --readyfile $SAUCE_READY_FILE_PATH \
      -i $BROWSERSTACK_LOCAL_ID \
      --no-ssl-bump-domains irc-ws.chat.twitch.tv,irc-ws-darklaunch.chat.twitch.tv,pubsub-edge.twitch.tv \
    "

    # Append Proxy Settings to BMP Options if BMP is true
    if [[ $BMP == true ]]; then
      BMP_FILE="/build/browsermob.js"

      # Sauce Labs requires a PAC File. Create that, pass in the BMP port.
      # https://wiki.saucelabs.com/display/DOCS/Sauce+Connect+with+a+Proxy+Setup#SauceConnectwithaProxySetup-BrowserMobProxyConfiguration
      cat > $BMP_FILE <<- EOM
function FindProxyForURL(url, host) {
    if (shExpMatch(host, "*.miso.saucelabs.com") ||
        shExpMatch(host, "saucelabs.com")) {
        // KGP and REST connections. Another proxy can also be specified.
        return "DIRECT";
    }

    // Test HTTP traffic, route it through the local BrowserMob proxy.
    return "bmp.us-west2.justin.tv:$BMP_PORT";
}
EOM

      # Add the PAC to the options string
      options+=" --pac file://$BMP_FILE"
    fi

    # Actually start Sauce Connect, with the compiled options
    daemon -n sauce_connect -o /var/log/sc.log -- /usr/local/bin/sc-4.4.2-linux/bin/sc $options

    # Wait for Sauce Connect to indicate ready. This is done by the readyfile. Loop until it is there.
    sauce_ready=false
    attempt=0
    max_attempt=60

    while [ $sauce_ready == false ]; do
      ((attempt+=1))

      if [ -f $SAUCE_READY_FILE_PATH ]; then
        echo "[INFO] Sauce Connect is Ready"
        sauce_ready=true
      else

        if [ $attempt -ge $max_attempt ]; then
          echo "[ERROR] Reached maximum attempts ($max_attempt) while waiting for Sauce to start. Aborting."
          echo "Log from Client: `cat /var/log/sc.log`"
          daemon -n sauce_connect --stop # Try to stop it just incase
          sleep 10 # Give some time to stop Sauce before exiting, just incase
          exit 1
        fi

        echo "[INFO] Waiting for Sauce Connect to be ready... Attempt #$attempt"
        sauce_ready=false
        sleep 3 # Lets pad some time between checks...
      fi
    done

  elif [[ $GRID_HOST == 'browserstack' ]]; then
    echo "[INFO] Starting BrowserStack Local"
    options="$BROWSERSTACK_KEY --force-local --local-identifier $BROWSERSTACK_LOCAL_ID --daemon start"

    # Append Proxy Settings to BMP Options if BMP is true
    if [[ $BMP == true ]]; then
      options+=" --force-proxy --local-proxy-host bmp.us-west2.justin.tv --local-proxy-port $BMP_PORT"
    fi

    /usr/bin/BrowserStackLocal $options
  fi
fi

###################
#  Run the Suite  #
###################

bundle exec rspec
# DO NOT ADD ANYTHING IN BETWEEN THIS AND bundle exec rspec
# Store the exit code of the test suite, to ensure we exit out of this script with that exit code (near the bottom)
# Ensures Sauce Connect logic will not override that exit code.
test_suite_exit_code=$?


########################################
#  Stop Sauce Connect (if applicable)  #
########################################

if [[ $LOCAL_TESTING == true && $GRID_HOST == 'saucelabs' ]]; then
  sauce_ready=true
  attempt=0
  max_attempt=60

  echo "[INFO] Stopping Sauce Connect"
  daemon -n sauce_connect --stop

  while [ $sauce_ready == true ]; do
    ((attempt+=1))

    if [ -f $SAUCE_READY_FILE_PATH ]; then

      if [ $attempt -ge $max_attempt ]; then
          echo "[ERROR] Reached maximum attempts ($max_attempt) while waiting for Sauce to stop. Aborting."
          daemon -n sauce_connect --stop # Try to stop it one more time just incase
          sleep 10 # Give some time to stop Sauce, just incase
          exit 1
      fi

      echo "[INFO] Waiting for Sauce Connect to shut down..."
      sauce_ready=true
      sleep 3 # Lets pad some time between checks...
    else
      echo "[INFO] Sauce Connect has shut down"
      sauce_ready=false
    fi
  done
fi

echo "[INFO] Preparing to exit initialization script. Test Suite exit code: $test_suite_exit_code"
# Exit the script with the exit code of the test suite.
exit $test_suite_exit_code
