provider "aws" {
  region  = "us-west-2"
  profile = "twitch-cape-qe-aws"
}

// Used for gathering Internal CIDRs
data "terraform_remote_state" "parent_account_sgs" {
  backend = "s3"
  config {
    profile = "twitch-cape-qe-aws"
    bucket = "tf-state-qe"
    key    = "tfstate/qe/terraform/security-groups"
    region = "us-west-2"
  }
}

// Create the Bucket
resource "aws_s3_bucket" "smoca_screenshot_bucket" {
  bucket = "smoca-screenshots"
  acl = "private"

  tags {
    Name = "smoca-screenshots"
    Owner = "reichsta"
  }
}

// Make it so that any IP within our internal network can access the bucket contents
// Also includes any instance trying to connect to it from the twitch-web-dev vpc
resource "aws_s3_bucket_policy" "screenshot_access" {
  bucket = "${aws_s3_bucket.smoca_screenshot_bucket.id}"
  policy =<<POLICY
{
  "Id": "Policy1506643744053",
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1506641993239",
      "Action": [
        "s3:GetObject"
      ],
      "Effect": "Allow",
      "Resource": "${aws_s3_bucket.smoca_screenshot_bucket.arn}/*",
      "Condition": {
        "StringEquals": {
          "aws:SourceVpc": "${data.terraform_remote_state.parent_account_vpc.vpc_id}"
        }
      },
      "Principal": "*"
    },
    {
      "Sid": "Stmt1506643742126",
      "Action": [
        "s3:GetObject"
      ],
      "Effect": "Allow",
      "Resource": "${aws_s3_bucket.smoca_screenshot_bucket.arn}/*",
      "Condition": {
        "IpAddress": {
          "aws:SourceIp": ["${join("\",\"", data.terraform_remote_state.parent_account_sgs.twitch_subnets)}"]
        }
      },
      "Principal": "*"
    }
  ]
}
POLICY
}

// IAM User for Smoca to Use

resource "aws_iam_user" "smoca-screenshot" {
  name = "smoca-screenshot"
}

// Add Read / Write / Destroy access to only the Smoca S3 Bucket Resource
resource "aws_iam_policy" "smoca-screenshot" {
  name        = "smoca_screenshot_access"
  path        = "/"
  description = "Smoca to Write and Read the smoca-screenshot bucket"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1506116661040",
      "Action": [
        "s3:DeleteObject",
        "s3:GetObject",
        "s3:PutObject"
      ],
      "Effect": "Allow",
      "Resource": "${aws_s3_bucket.smoca_screenshot_bucket.arn}/*"
    }
  ]
}
EOF
}

// Attach that policy to the smoca user
resource "aws_iam_user_policy_attachment" "smoca-screenshot" {
  user       = "${aws_iam_user.smoca-screenshot.name}"
  policy_arn = "${aws_iam_policy.smoca-screenshot.arn}"
}
