#!/usr/bin/env python
from collections import namedtuple
import json
import os
import os.path
import sys

# GoSpec is a specifier for a downloadable (.tar.gz) release of Go.
GoSpec = namedtuple("GoSpec", "version os arch sha sha_algorithm")

# List of versions of go to generate manta files for
versions = [
    GoSpec("1.7.1", "linux", "amd64", "43ad621c9b014cde8db17393dc108378d37bc853aa351a6c74bf6432c1bbd182", "256"),
    GoSpec("1.8.3", "linux", "amd64", "1862f4c3d3907e59b04a757cfda0ea7aa9ef39274af99a784f5be843c80c6772", "256"),
    GoSpec("1.9.2", "linux", "amd64", "de874549d9a8d8d8062be05808509c09a88a248e77ec14eb77453530829ac02b", "256"),
    GoSpec("1.10beta2", "linux", "amd64", "ab3abb7d731dd5ac7a06d5d5e64ef19946f57d4ce34555d262a87b8899901a93", "256"),
]


def go_identifier_string(spec):
    return "go{v}.{os}-{arch}".format(v=spec.version, os=spec.os, arch=spec.arch)


def echolog(spec, msg):
    return "echo '{spec}: {msg}'".format(spec=go_identifier_string(spec), msg=msg)


def generate(gospec):
    ''' Create a dictionary of Manta build instructions. '''
    return {
        "image": "ubuntu:xenial",
        "mount": "/go/src/code.justin.tv/common/chitin",
        "env": [
            "GOPATH=/go/src/code.justin.tv/common/chitin/Godeps/_workspace:/go",
            "PATH=/usr/local/go/bin:$PATH"
        ],
        "setup": [
            download_go(gospec),
            "tar -C /usr/local -xzf go.tar.gz"
        ],
        "build": [
            echolog(gospec, "Checking that code matches go fmt"),
            # This is a little wordy, but go fmt doesn't exit with a
            # nonzero code - we need to check whether it printed any
            # output.
            """nonfmt=$(go fmt ./...)
            if [ -n "${nonfmt}" ]; then
                echo "${nonfmt}" | sed -e 's/^/non-gofmt-code: /'
                false
            fi""",

            echolog(gospec, "Running go vet"),
            go_vet_conditionally(gospec),

            echolog(gospec, "Running go test"),
            "go test ./...",

            echolog(gospec, "Running go test -race"),
            "go test -race ./...",
        ],
    }


def download_go(spec):
    ''' Generate a command which will download go at a particular version for an os, and check its checksum. '''
    template = "wget -q -O go.tar.gz http://storage.googleapis.com/golang/{pkg}.tar.gz && echo '{sha}  go.tar.gz' | sha{sha_alg}sum --check"
    return template.format(
        pkg=go_identifier_string(spec),
        sha=spec.sha,
        sha_alg=spec.sha_algorithm,
    )


def go_vet_conditionally(spec):
    # go vet doesn't ship with go 1.4.3 :(
    # see https://github.com/golang/go/issues/12883
    if spec.version == '1.4.3':
        return "echo 'Skipping go vet, since it does not ship with go 1.4.3'"
    else:
        return "go vet ./..."



def manta_filename(spec):
    return "mantas/manta.{pkg}.json".format(pkg=go_identifier_string(spec))


def write_manta_file(spec):
    with open(manta_filename(spec), "w") as f:
        json.dump(generate(spec), f, indent=2)


def write_jenkins_groovy(specs):
    # Instead of using .format here, I've just concatenated strings to
    # avoid the gnarliness of escaping all the curly braces.

    steps = ["shell 'manta -proxy -f _build/{0}'".format(manta_filename(s)) for s in specs]
    steps = "\n".join(steps)

    jenkins_groovy = """
    job {
	name "common-chitin"
	using 'TEMPLATE-autobuild'
	scm {
            git {
		remote {
			github 'common/chitin', 'ssh', 'git-aws.internal.justin.tv'
			credentials 'git-aws-read-key'
		}
		clean true
	    }
	}
	steps {
""" + steps + """
	}
    }

    job {
	name 'common-chitin-deploy'
	using 'TEMPLATE-deploy'
	steps {
	}
    }
    """

    with open("../jenkins.groovy", "w") as f:
        f.write(jenkins_groovy)


def abort(msg):
    print "FATAL: " + msg
    sys.exit(1)


def ensure_working_directory():
    '''Abort if the script is not being run from its directory on disk '''
    cwd = os.getcwd()
    file_dir = os.path.abspath(os.path.dirname(__file__))
    if not cwd == file_dir:
        abort("""
mantamaker.py should only be run from within its directory on disk:
    {want}
It seems to be running from here though:
    {have}""".format(want=file_dir, have=cwd))


def main():
    ensure_working_directory()
    for spec in versions:
        write_manta_file(spec)
    write_jenkins_groovy(versions)


if __name__ == '__main__':
    main()
