package main

import (
	"bytes"
	"context"
	"crypto/tls"
	"flag"
	"fmt"
	"html/template"
	"io"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"os"
	"os/exec"
	"path/filepath"
	"time"

	"code.justin.tv/common/chitin"
	_ "code.justin.tv/common/golibs/bininfo"
	"code.justin.tv/release/trace/api/report_v1"
	"code.justin.tv/release/trace/persistent"
	"golang.org/x/net/trace"
)

const (
	timeFormat = "2006-01-02-15-04-05.000000"
	// How long to read the stream from kafka:
	reportLength = 30 * time.Second

	defaultDBDir = "db"
)

func main() {
	var (
		period    = flag.Duration("period", 10*time.Second, "Delay between reporting cycles")
		reportDir = flag.String("reports", "", "Path to reports directory")
		cmd       = flag.String("cmd", "/bin/true", "Path to report command")
		httpAddr  = flag.String("http", ":11443", "Address on which to expose HTTP listener")
	)
	flag.Parse()

	err := chitin.ExperimentalTraceProcessOptIn()
	if err != nil {
		log.Fatalf("trace enable err=%q", err)
	}

	mux := http.NewServeMux()
	mux.Handle("/debug/", http.DefaultServeMux)
	mux.HandleFunc("/debug/http", handleDebugHTTP)
	mux.HandleFunc("/debug/running", handleDebugRunning)

	index, err := persistent.NewFilesystemReportIndex(http.Dir(*reportDir))
	if err != nil {
		log.Fatalf("persistent filesystem err=%q", err)
	}

	apiSrv := &reportServer{
		index:      index,
		htmlPrefix: "/v1/html/",
	}
	mux.Handle(report_v1.ReportsPathPrefix, report_v1.NewReportsServer(apiSrv, apiSrv.twirpHooks(), chitin.Context))
	mux.Handle(apiSrv.htmlPrefix, apiSrv)

	var handler http.Handler = &browserHandler{
		reports:  apiSrv,
		backfill: mux,
	}

	prev := trace.AuthRequest
	trace.AuthRequest = func(req *http.Request) (any, sensitive bool) {
		any, sensitive = prev(req)
		any = true
		return any, sensitive
	}
	ctx := context.Background()

	l, err := net.Listen("tcp", *httpAddr)
	if err != nil {
		log.Fatalf("listen err=%q", err)
	}
	cert, err := genCert()
	if err != nil {
		log.Fatalf("cert err=%q", err)
	}

	config := &tls.Config{
		Certificates: []tls.Certificate{cert},
		NextProtos:   []string{"h2", "http/1.1"},
	}
	l = tls.NewListener(l, config)
	srv := http.Server{
		Handler: chitin.Handler(handler),
	}
	go func() {
		err := srv.Serve(l)
		if err != nil {
			log.Fatalf("http serve err=%q", err)
		}
	}()

	// we'll run reports periodically
	tick := time.NewTimer(*period)
	defer tick.Stop()
	// the first run should happen immediately
	tick.Reset(1 * time.Nanosecond)

	for {
		// time to go!
		<-tick.C
		// TODO: jitter
		tick.Reset(*period)

		task := reportTask{
			reportDir: *reportDir,
			reportCmd: *cmd,
		}
		err := task.run(ctx)
		if err != nil {
			log.Printf("report err=%q", err)
		}
	}
}

type reportTask struct {
	reportDir string
	reportCmd string
}

func (task *reportTask) run(ctx context.Context) (err error) {
	log.Printf("report-state=%q", "begin")
	defer log.Printf("report-state=%q", "end")

	// Put the result in a timestamped directory
	timestamp := time.Now().Format(timeFormat)
	destination := filepath.Join(task.reportDir, timestamp)
	log.Printf("destination=%q", destination)

	tr := trace.New("function", "reportTask.run")
	defer tr.Finish()

	tr.LazyPrintf("timestamp=%q", timestamp)
	ctx = trace.NewContext(ctx, tr)
	defer func() {
		if err != nil {
			tr.SetError()
			tr.LazyPrintf("err=%q", err)
		}
	}()

	if _, err := os.Stat(destination); err != nil {
		if !os.IsNotExist(err) {
			return err
		}
	} else {
		return fmt.Errorf("report destination already exists: %v", destination)
	}

	err = os.MkdirAll(destination, 0755)
	if err != nil {
		return err
	}

	outdir := filepath.Join(destination, "report")
	dbdir := filepath.Join(destination, defaultDBDir)

	// nginx is serving from task.reportDir, so all URLs should be
	// relative to that path
	urlPrefix, err := filepath.Rel(task.reportDir, outdir)
	if err != nil {
		return err
	}
	urlPrefix = "/" + urlPrefix

	// Stopgap for an issue where failed Kinesis requests can result in the
	// txreport task stalling indefinitely. Killing the process after 30
	// minutes is enough to allow normal reports to run successfully (Dec 2016
	// reports take around two minutes to process 30 seconds of data) while
	// not entirely hiding the problem.
	//
	// https://git-aws.internal.justin.tv/release/trace/issues/90
	reportCtx, reportCancel := context.WithTimeout(ctx, 30*time.Minute)
	defer reportCancel()

	cmd := exec.CommandContext(reportCtx, task.reportCmd,
		"-cpuprofile", filepath.Join(destination, "cpu.out"),
		"-flush", reportLength.String(),
		"-outdir", outdir,
		"-urlprefix", urlPrefix,
		"-db", dbdir,
	)
	setOptions(cmd)

	stdout, err := os.Create(filepath.Join(destination, "stdout.txt"))
	if err != nil {
		return err
	}
	defer func(c io.Closer) {
		err := c.Close()
		if err != nil {
			log.Printf("close err=%q", err)
		}
	}(stdout)
	stderr, err := os.Create(filepath.Join(destination, "stderr.txt"))
	if err != nil {
		return err
	}
	defer func(c io.Closer) {
		err := c.Close()
		if err != nil {
			log.Printf("close err=%q", err)
		}
	}(stderr)
	cmd.Stdout, cmd.Stderr = stdout, stderr

	err = cmd.Run()
	if err != nil {
		return err
	}

	// TODO: we get the location of the pprof executable as relative to this
	// runreport executable .. is this too dirty?

	// TODO: dereference report binary once at the start of this function so
	// we'll run pprof against the binary that we ran, even if there's a
	// concurrent deploy.

	ourBin, err := filepath.Abs(os.Args[0])
	if err != nil {
		return err
	}
	log.Printf("ourbin=%q", ourBin)
	pprofBin := filepath.Join(filepath.Dir(ourBin), "../tool/pprof")
	log.Printf("pprofbin=%q", pprofBin)

	pprof := exec.CommandContext(ctx, pprofBin,
		"-proto", "-output", filepath.Join(destination, "cpu.pb.gz"),
		cmd.Path, filepath.Join(destination, "cpu.out"))
	setOptions(pprof)
	log.Printf("tmpdir=%q", os.TempDir())
	pprof.Env = []string{"PPROF_TMPDIR=" + os.TempDir()}
	buf, err := pprof.CombinedOutput()
	log.Printf("pprof output=%q", buf)
	if err != nil {
		return err
	}

	err = task.writeIndex(filepath.ToSlash(filepath.Join(timestamp, "report")))
	if err != nil {
		return err
	}

	return nil
}

// writeIndex updates the "index.html" redirect page
func (task *reportTask) writeIndex(uri string) error {
	var buf bytes.Buffer

	template.Must(template.New("").Parse(`
<!doctype html>
<html lang="en">

<head>
  <meta charset="utf-8">
  <meta http-equiv="refresh" content="0; url={{.}}" />
  <title>Current Trace Report</title>
</head>

<body>
  <p>The current Trace report is available at <a href="{{.}}">{{.}}</a>.</p>
</body>
</html>
`[1:])).Execute(&buf, uri+"/")

	finalName := filepath.Join(task.reportDir, "index.html")

	tmpFile, err := ioutil.TempFile(task.reportDir, "index")
	if err != nil {
		return err
	}

	defer func() {
		err := os.Remove(tmpFile.Name())
		if err != nil {
			// best-effort cleanup, ignore any error
		}
	}()

	_, err = io.Copy(tmpFile, &buf)
	cerr := tmpFile.Close()
	if err != nil {
		return err
	}
	if cerr != nil {
		return cerr
	}

	err = os.Rename(tmpFile.Name(), finalName)
	if err != nil {
		return err
	}

	return nil
}

var osSetOptions func(cmd *exec.Cmd)

func setOptions(cmd *exec.Cmd) {
	if osSetOptions != nil {
		osSetOptions(cmd)
	}
}
