package main

import (
	"errors"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"
	"time"

	"code.justin.tv/release/trace/api"
	"code.justin.tv/release/trace/api/report_v1"
	"github.com/golang/protobuf/proto"
	"github.com/golang/protobuf/ptypes"
	netcontext "golang.org/x/net/context"
)

func TestRouteGet(t *testing.T) {
	br := &browserHandler{
		reports: &testReports{},
	}

	testcase := func(path string, contentType string, contentFragment string) func(t *testing.T) {
		return func(t *testing.T) {
			srv := httptest.NewServer(br)
			defer srv.Close()

			resp, err := http.Get(srv.URL + path)
			if err != nil {
				t.Fatalf("http.Get; err = %v", err)
			}
			defer resp.Body.Close()

			if have, want := resp.StatusCode, http.StatusOK; have != want {
				t.Errorf("http.Get.StatusCode; %d != %d", have, want)
			}
			if have, want := resp.Header.Get("Content-Type"), contentType; !strings.Contains(have, want) {
				t.Errorf("http.Get.Header.Content-Type; %q does not contain %q", have, want)
			}
			body, err := ioutil.ReadAll(resp.Body)
			if err != nil {
				t.Fatalf("ioutil.ReadAll; err = %v", err)
			}
			if have, want := string(body), contentFragment; !strings.Contains(have, want) {
				t.Errorf("http.Get.Body; %q does not contain %q", have, want)
			}
		}
	}

	t.Run("css", testcase("/2017-01-01/report/css/style.css", "text/css", "padding-left"))
	t.Run("js", testcase("/2017-01-01/report/js/tx.js", "application/js", "hasOwnProperty"))

	t.Run("latest", testcase("/latest", "text/html", "2001-09-09-01-46-40.000000/report"))

	t.Run("program list", testcase("/2001-09-09-01-46-40.000000/report",
		"text/html", `<a href="/2001-09-09-01-46-40.000000/report/bin/code.justin.tv/web/web">`))
	t.Run("program detail", testcase("/2001-09-09-01-46-40.000000/report/bin/code.justin.tv/web/web",
		"text/html", `<li>code.justin.tv/web/streams-api</li>`))
	t.Run("transaction display", testcase("/2001-09-09-01-46-40.000000/report/tx",
		"text/html", `<script type="text/javascript" src="/2001-09-09-01-46-40.000000/report/js/tx.js"></script>`))
	t.Run("transaction content", testcase("/2001-09-09-01-46-40.000000/report/tx/2a3a0dad5b22df163624a8fb54e2a50a",
		"text/plain", "LookupTransaction unimplemented"))
}

// txString is a string-formatted protobuf message representing transaction
// 2a3a0dad5b22df163624a8fb54e2a50a.
var txString = `
transaction_id: 1648073765781584426
transaction_id: 767268166163768374
root: <
  subcalls: <
    subcalls: <
      path: 0
      path: 0
      svc: <
      >
      params: <
        sql: <
          dbname: "site_justintv_prod"
          dbuser: "site_02"
          stripped_query: "SELECT \"user_channel_properties\".* FROM \"user_channel_properties\" WHERE \"user_channel_properties\".\"user_id\" = ? LIMIT ?"
        >
      >
      client_timestamps:<time:1460750722581595347 kind:REQUEST_HEAD_PREPARED >
      client_timestamps:<time:1460750722583736838 kind:RESPONSE_HEAD_RECEIVED >
      request_sent_to: "localhost:12006"
    >
    path: 0
    svc:<name:"code.justin.tv/web/web" host:"rails-app-aws-bacfcd60.prod.us-west2.justin.tv" pid:48007 peer:"10.194.237.34:0" >
    params:<http:<status:200 method:GET route:"kraken/channels#show" > >
    client_timestamps:<time:1460750722567297143 kind:REQUEST_HEAD_PREPARED >
    client_timestamps:<time:1460750722591126322 kind:RESPONSE_HEAD_RECEIVED >
    service_timestamps:<time:1460750722568853128 kind:REQUEST_BODY_RECEIVED >
    service_timestamps:<time:1460750722588784573 kind:RESPONSE_HEAD_PREPARED >
    request_sent_to: "internal-internal-nginx-1811164580.us-west-2.elb.amazonaws.com:80"
  >
  subcalls: <
    path: 1
    svc:<name:"code.justin.tv/web/jax/api" host:"jax-aws-dd5b7e05.prod.us-west2.justin.tv" pid:29509 peer:"127.0.0.1:19912" >
    params:<http:<status:200 method:GET > >
    client_timestamps:<time:1460750722591600709 kind:REQUEST_HEAD_PREPARED >
    client_timestamps:<time:1460750722600834711 kind:RESPONSE_HEAD_RECEIVED >
    service_timestamps:<time:1460750722594356363 kind:REQUEST_HEAD_RECEIVED >
    service_timestamps:<time:1460750722599393056 kind:RESPONSE_HEAD_PREPARED >
    request_sent_to: "jax-internal-production.us-west2.justin.tv:80"
  >
  svc:<name:"code.justin.tv/web/streams-api" host:"streams-api-25f344e1.prod.us-west2.justin.tv" pid:56284 peer:"10.192.71.21:53097" >
  params:<http:<status:200 method:GET route:"stream" > >
  service_timestamps:<time:1460750722567161315 kind:REQUEST_HEAD_RECEIVED >
  service_timestamps:<time:1460750722600967500 kind:RESPONSE_HEAD_PREPARED >
>
client: <>
`

// TODO: build report from *api.Transaction iterator

type testReports struct {
}

var testReportTime = time.Unix(1e9, 0).UTC()

func (tr *testReports) ListReports(ctx netcontext.Context, req *report_v1.ListReportsRequest) (*report_v1.ListReportsResponse, error) {
	tsp, err := ptypes.TimestampProto(testReportTime)
	if err != nil {
		return nil, err
	}

	return &report_v1.ListReportsResponse{
		Report: []*report_v1.Report{
			&report_v1.Report{Id: testReportTime.Format(timeFormat), Modified: tsp},
		},
	}, nil
}

func (tr *testReports) LookupTransaction(ctx netcontext.Context, req *report_v1.LookupTransactionRequest) (*report_v1.LookupTransactionResponse, error) {
	if req.ReportId != testReportTime.Format(timeFormat) {
		return nil, errors.New("report not found")
	}
	if req.TransactionId != "2a3a0dad5b22df163624a8fb54e2a50a" {
		return nil, errors.New("transaction not found")
	}

	var tx api.Transaction
	err := proto.UnmarshalText(txString, &tx)
	if err != nil {
		return nil, err
	}

	return &report_v1.LookupTransactionResponse{Transaction: &tx}, nil
}

func (tr *testReports) GetProgramReport(ctx netcontext.Context, req *report_v1.GetProgramReportRequest) (*report_v1.GetProgramReportResponse, error) {
	if req.ReportId != testReportTime.Format(timeFormat) {
		return nil, errors.New("report not found")
	}
	if req.GetProgram().GetName() != "code.justin.tv/web/web" {
		return nil, errors.New("program not found")
	}

	callRef := &report_v1.CallRef{TransactionId: "", Path: []uint32{}}
	callSummary := &report_v1.SubtreeSummary{
		Call: &report_v1.CallSummary{
			Ref:            callRef,
			ServerDuration: ptypes.DurationProto(38 * time.Millisecond),
		},
		SubtreeSize:  2,
		SubtreeDepth: 2,
	}

	return &report_v1.GetProgramReportResponse{
		Content: &report_v1.ProgramReport{
			Program: &report_v1.ProgramRef{Name: "code.justin.tv/web/web"},
			Noteworthy: &report_v1.NoteworthyTransactions{
				Deepest: callSummary, Largest: callSummary, Slowest: callSummary,
			},
			Signature: []*report_v1.CallSignatureSummary{
				{
					Method: nil,
					Dependencies: []*report_v1.ProgramRef{
						{Name: "code.justin.tv/web/jax/api"},
						{Name: "code.justin.tv/web/streams-api"},
					},
					TransitiveDependencies: []*report_v1.ProgramRef{
						{Name: "code.justin.tv/web/jax/api"},
						{Name: "code.justin.tv/web/streams-api"},
					},
					Examples: []*report_v1.CallRef{callRef},
				},
			},
			RootServerTimingDistribution: &report_v1.Distribution{},
		},
	}, nil
}

func (tr *testReports) ListProgramsInReport(ctx netcontext.Context, req *report_v1.ListProgramsInReportRequest) (*report_v1.ListProgramsInReportResponse, error) {
	if req.ReportId != testReportTime.Format(timeFormat) {
		return nil, errors.New("report not found")
	}
	return &report_v1.ListProgramsInReportResponse{Programs: []*report_v1.ProgramRef{{Name: "code.justin.tv/web/web"}}}, nil
}
