package htmlreport

import (
	"math"
	"time"

	"code.justin.tv/release/trace/api/report_v1"
)

func serviceVegaSpec(dist *report_v1.Distribution) interface{} {

	type vegadata struct {
		Name   string        `json:"name"`
		Values []interface{} `json:"values"`
	}
	type spec struct {
		Width   int `json:"width"`
		Height  int `json:"height"`
		Padding struct {
			Top    int `json:"top"`
			Bottom int `json:"bottom"`
			Left   int `json:"left"`
			Right  int `json:"right"`
		} `json:"padding"`
		Data   []vegadata               `json:"data"`
		Scales []map[string]interface{} `json:"scales"`
		Axes   []map[string]interface{} `json:"axes"`
		Marks  []map[string]interface{} `json:"marks"`
	}

	var body spec
	body.Width = 800
	body.Height = 500
	body.Padding.Top = 10
	body.Padding.Bottom = 20
	body.Padding.Left = 60
	body.Padding.Right = 10
	body.Scales = []map[string]interface{}{
		{
			"name": "x", "type": "log", "range": "width", "nice": true,
			"domain": map[string]string{"data": "table", "field": "data.x"},
		},
		{
			"name": "y", "type": "log", "range": "height", "nice": true,
			"domain":    map[string]string{"data": "table", "field": "data.y"},
			"domainMin": 0.1,
		},
	}
	body.Axes = []map[string]interface{}{
		{"type": "x", "scale": "x"},
		{"type": "y", "scale": "y"},
	}
	body.Marks = []map[string]interface{}{
		{
			"type": "rect",
			"from": map[string]string{"data": "table"},
			"properties": map[string]interface{}{
				"enter": map[string]interface{}{
					"x":  map[string]interface{}{"scale": "x", "field": "data.x"},
					"x2": map[string]interface{}{"scale": "x", "field": "data.x2"},
					"y":  map[string]interface{}{"scale": "y", "field": "data.y"},
					"y2": map[string]interface{}{"scale": "y", "value": 0.1},
				},
				"update": map[string]interface{}{
					"fill": map[string]interface{}{"value": "steelblue"},
				},
				"hover": map[string]interface{}{
					"fill": map[string]interface{}{"value": "red"},
				},
			},
		},
	}

	data := []interface{}{}

	opts := dist.GetBucketOptions().GetExponentialBuckets()
	if opts == nil {
		return nil
	}

	for i, n := range dist.BucketCounts {
		var min, max float64

		if i > 0 {
			min = opts.Scale * math.Pow(opts.GrowthFactor, float64(i-1))
		} else {
			// log scale, can't start at zero
			min = 1
		}
		max = opts.Scale * math.Pow(opts.GrowthFactor, float64(i))

		// The final bucket is different since it can represent arbitrarily-
		// large data points. We start by making it cover the same log-scale
		// range as a normal bucket.
		if i == int(opts.NumFiniteBuckets)+1 {
			// If the size we've estimated for the final bucket doesn't extend
			// far enough to capture all known data points, increase its size
			// appropriately.
			if max < dist.GetRange().GetMax() {
				max = dist.GetRange().GetMax()
			}
		}

		type xy struct {
			X  int64 `json:"x"`
			X2 int64 `json:"x2"`
			Y  int64 `json:"y"`
		}
		data = append(data, xy{
			X:  int64(min / time.Nanosecond.Seconds()),
			X2: int64(max / time.Nanosecond.Seconds()),
			Y:  n,
		})
	}

	body.Data = []vegadata{
		{
			Name:   "table",
			Values: data,
		},
	}

	return body
}
