package persistent

import (
	"context"
	"io"
	"io/ioutil"
	"net/http"
	"path"
	"time"

	thistrace "code.justin.tv/release/trace"
	"code.justin.tv/release/trace/api"
	"code.justin.tv/release/trace/api/report_v1"
	"github.com/pkg/errors"
	"github.com/syndtr/goleveldb/leveldb"
	"github.com/syndtr/goleveldb/leveldb/opt"
	"github.com/syndtr/goleveldb/leveldb/storage"
	"github.com/syndtr/goleveldb/leveldb/util"
	"golang.org/x/net/trace"
)

func NewFilesystemReportIndex(root http.FileSystem) (*FilesystemReportIndex, error) {
	return &FilesystemReportIndex{
		root: root,
	}, nil
}

type FilesystemReportIndex struct {
	root http.FileSystem
}

var _ ReportIndex = (*FilesystemReportIndex)(nil)

func (fs *FilesystemReportIndex) List(ctx context.Context) ([]Report, error) {
	dir, err := fs.root.Open(".")
	if err != nil {
		return nil, err
	}
	defer dir.Close()

	infos, err := dir.Readdir(0)
	if err != nil {
		return nil, err
	}

	var reports []Report
	for _, info := range infos {
		reports = append(reports, &filesystemReport{
			root: fs.root,
			id:   info.Name(),
			mod:  info.ModTime(),
		})
	}

	return reports, nil
}

type filesystemReport struct {
	root http.FileSystem
	id   string
	mod  time.Time
}

func (rep *filesystemReport) ID() string {
	return rep.id
}

func (rep *filesystemReport) ModTime() time.Time {
	return rep.mod
}

func (rep *filesystemReport) Pluck(ctx context.Context, txid *thistrace.ID) (*api.Transaction, error) {
	db, err := openReadonly(ctx, rep.root, path.Join(".", rep.id, "db"))
	if err != nil {
		return nil, err
	}
	defer db.Close()

	tx, err := db.ReadTransaction(txid)
	if err != nil {
		return nil, err
	}

	return tx, nil
}

func (rep *filesystemReport) ListServers(ctx context.Context) ([]string, error) {
	db, err := openReadonly(ctx, rep.root, path.Join(".", rep.id, "db"))
	if err != nil {
		return nil, err
	}
	defer db.Close()

	servers, err := db.ListServers()
	if err != nil {
		return nil, err
	}

	return servers, nil
}

func (rep *filesystemReport) Content(ctx context.Context, serverName string) (*report_v1.ProgramReport, error) {
	db, err := openReadonly(ctx, rep.root, path.Join(".", rep.id, "db"))
	if err != nil {
		return nil, err
	}
	defer db.Close()

	content, err := db.ReadProgramReport(serverName)
	if err != nil {
		return nil, err
	}

	return content, nil
}

func openReadonly(ctx context.Context, fs http.FileSystem, path string) (*DB, error) {
	db, err := leveldb.Open(&httpFileSystemStorage{ctx: ctx, fs: fs, path: path}, &opt.Options{ReadOnly: true})
	if err != nil {
		return nil, err
	}

	return &DB{
		ldb: db,
	}, nil
}

type httpFileSystemStorage struct {
	ctx  context.Context
	fs   http.FileSystem
	path string
}

func (stor *httpFileSystemStorage) logf(format string, a ...interface{}) {
	tr, ok := trace.FromContext(stor.ctx)
	if !ok {
		return
	}
	tr.LazyPrintf(format, a...)
}

// GetMeta returns the name of the current manifest file.
//
// It is part of the github.com/syndtr/goleveldb/leveldb/storage.Storage
// interface.
func (stor *httpFileSystemStorage) GetMeta() (storage.FileDesc, error) {
	stor.logf("db cmd=%s", "GetMeta")

	var fd storage.FileDesc

	f, err := stor.fs.Open(path.Join(stor.path, "CURRENT"))
	if err != nil {
		stor.logf("db err=%q", err)
		return fd, err
	}
	defer f.Close()

	b, err := ioutil.ReadAll(io.LimitReader(f, 1<<10))
	if err != nil {
		stor.logf("db err=%q", err)
		return fd, err
	}

	fd, ok := fsParseName(string(b))
	if !ok {
		stor.logf("db err=%q", err)
		return fd, errors.New("invalid manifest file name")
	}

	return fd, nil
}

// List returns a list of all database files of the given type.
//
// It is part of the github.com/syndtr/goleveldb/leveldb/storage.Storage
// interface.
func (stor *httpFileSystemStorage) List(ft storage.FileType) ([]storage.FileDesc, error) {
	stor.logf("db cmd=%s", "List")

	dir, err := stor.fs.Open(stor.path)
	if err != nil {
		stor.logf("db err=%q", err)
		return nil, err
	}
	defer dir.Close()

	infos, err := dir.Readdir(0)
	if err != nil {
		stor.logf("db err=%q", err)
		return nil, err
	}

	var fds []storage.FileDesc
	for _, info := range infos {
		fd, ok := fsParseName(info.Name())
		if !ok || fd.Type != ft {
			continue
		}
		fds = append(fds, fd)
	}

	return fds, nil
}

// Log records a message to the database's informational log (or ignores it).
//
// It is part of the github.com/syndtr/goleveldb/leveldb/storage.Storage
// interface.
func (stor *httpFileSystemStorage) Log(str string) {
	stor.logf("db cmd=%s message=%q", "Log", str)
}

// Open opens a specified database file for reading.
//
// It is part of the github.com/syndtr/goleveldb/leveldb/storage.Storage
// interface.
func (stor *httpFileSystemStorage) Open(fd storage.FileDesc) (storage.Reader, error) {
	stor.logf("db cmd=%s", "Open")

	f, err := stor.fs.Open(path.Join(stor.path, fd.String()))
	if err != nil {
		return nil, err
	}

	return &httpFileSystemFile{
		stor: stor,
		fd:   fd,
		f:    f,
	}, nil
}

// Close cleans up process state associated with an open database.
//
// It is part of the github.com/syndtr/goleveldb/leveldb/storage.Storage
// interface.
func (stor *httpFileSystemStorage) Close() error {
	stor.logf("db cmd=%s", "Close")
	return nil
}

// Lock is supposed to protect against concurrent access to a database, but
// this implementation does nothing.
//
// It is part of the github.com/syndtr/goleveldb/leveldb/storage.Storage
// interface.
func (stor *httpFileSystemStorage) Lock() (storage.Lock, error) {
	stor.logf("db cmd=%s", "Lock")
	return util.NoopReleaser{}, nil
}

// Create opens a specified database file for writing.
//
// It is part of the github.com/syndtr/goleveldb/leveldb/storage.Storage
// interface.
func (stor *httpFileSystemStorage) Create(fd storage.FileDesc) (storage.Writer, error) {
	stor.logf("db cmd=%s", "Create")
	return nil, errors.New("database is in read-only mode")
}

// Remove deletes a specified database file.
//
// It is part of the github.com/syndtr/goleveldb/leveldb/storage.Storage
// interface.
func (stor *httpFileSystemStorage) Remove(fd storage.FileDesc) error {
	stor.logf("db cmd=%s", "Remove")
	return errors.New("database is in read-only mode")
}

// Rename changes the name of a specified database file.
//
// It is part of the github.com/syndtr/goleveldb/leveldb/storage.Storage
// interface.
func (stor *httpFileSystemStorage) Rename(oldfd, newfd storage.FileDesc) error {
	stor.logf("db cmd=%s", "Rename")
	return errors.New("database is in read-only mode")
}

// SetMeta changes the name of the currently relevant manifest file.
//
// It is part of the github.com/syndtr/goleveldb/leveldb/storage.Storage
// interface.
func (stor *httpFileSystemStorage) SetMeta(fd storage.FileDesc) error {
	stor.logf("db cmd=%s", "SetMeta")
	return errors.New("database is in read-only mode")
}

type httpFileSystemFile struct {
	stor *httpFileSystemStorage
	fd   storage.FileDesc
	f    http.File
}

func (file *httpFileSystemFile) Read(p []byte) (int, error) {
	return file.f.Read(p)
}
func (file *httpFileSystemFile) Seek(offset int64, whence int) (int64, error) {
	return file.f.Seek(offset, whence)
}
func (file *httpFileSystemFile) Close() error {
	return file.f.Close()
}
func (file *httpFileSystemFile) ReadAt(p []byte, off int64) (int, error) {
	f, err := file.stor.fs.Open(path.Join(file.stor.path, file.fd.String()))
	if err != nil {
		return 0, err
	}
	defer f.Close()

	_, err = f.Seek(off, io.SeekStart)
	if err != nil {
		return 0, err
	}

	return f.Read(p)
}
