package combiner

import (
	"fmt"
	"runtime"
	"testing"
	"time"
)

// Record mock
type record []byte

func fakeRecord(size int) Record {
	r := record(make([]byte, size))
	return &r
}

func (r *record) Key() int {
	return -42
}

func (r *record) NewSet() RecordSet {
	return &recordSet{}
}

// RecordSet mock
type recordSet []*record

func (rs *recordSet) Push(r Record) {
	*rs = append(*rs, r.(*record))
}

func (rs *recordSet) Pop() Record {
	var r *record
	a := *rs
	r, *rs = a[len(a)-1], a[:len(a)-1]
	return r
}

func (rs *recordSet) Size() int {
	return len(rs.Encode())
}

func (rs *recordSet) Encode() []byte {
	d := []byte{}
	for _, r := range *rs {
		d = append(d, ([]byte)(*r)...)
	}
	return d
}

func TestRecordSetMock(t *testing.T) {
	r := fakeRecord(100)
	rs := r.NewSet()

	assertSize(t, rs, 0)
	rs.Push(r)
	assertSize(t, rs, 100)
	popped := rs.Pop()
	assertSize(t, rs, 0)
	if popped != r {
		t.Fatal("Popped record did not equal pushed record")
	}
}

const assertionTimeout = 10 * time.Millisecond

// utility to print calling location for assertions
func caller() string {
	calldepth := 2
	_, file, line, ok := runtime.Caller(calldepth)
	if !ok {
		return "???"
	}
	return fmt.Sprintf("%s:%d", file, line)
}

// Helper methods
func assertEmpty(t *testing.T, ch <-chan *Blob) {
	select {
	case <-ch:
		t.Fatalf("%s: Channel was not empty", caller())
	case <-time.After(assertionTimeout):
	}
}

func assertAvailable(t *testing.T, ch <-chan *Blob) *Blob {
	select {
	case b, ok := <-ch:
		if !ok {
			t.Fatalf("%s: Channel was closed", caller())
		}
		return b
	case <-time.After(assertionTimeout):
		t.Fatalf("%s: Channel was empty", caller())
	}
	return nil
}

func assertNoError(t *testing.T, ch <-chan error) {
	select {
	case err, ok := <-ch:
		if !ok {
			return
		}
		t.Fatalf("%s: Unexpected error: %q", caller(), err)
	case <-time.After(assertionTimeout):
	}
}

func assertError(t *testing.T, ch <-chan error) {
	select {
	case _, ok := <-ch:
		if !ok {
			t.Fatalf("%s: Error channel was closed", caller())
		}
	case <-time.After(assertionTimeout):
		t.Fatalf("%s: Unexpected lack of error", caller())
	}
}

func assertSize(t *testing.T, rs RecordSet, size int) {
	if rs.Size() != size {
		t.Fatalf("%s: Incorrect recordSet size", caller())
	}
}
