package main

import (
	"fmt"
	"os"
	"path/filepath"
	"strconv"
	"strings"

	"github.com/beevik/etree"
	"github.com/urfave/cli"
)

var (
	testReportName = "test_report.xml"
	testReportPath = ""
)

// Represents an XML file we were passed.
type JUnitXMLFile struct {
	rootElement *etree.Element
	filePath    string
	testType    string
	numTests    int
	numFails    int
	numErrors   int
	duration    float64
}

func init() {
	cmd := cli.Command{
		Name:        "get-tests",
		Usage:       fmt.Sprintf("Merges multiple JUnit files into '%s'", testReportName),
		Description: fmt.Sprintf("Merges a list of JUnit files into a single file named '%s'.  You may also provided\n   directories as arguments - these will be recursively scanned for files ending in '.xml' and\n   merge them.\n\n   If an error is encountered with a file during the merge, then that file won't be merged.\n\n   You may also apply a flag to an arg to define the tpye of tests.  If you don't apply a\n   flag, then the type is assumed to be 'UNIT'.", testReportName),
		Action:      getTests,
		Flags: []cli.Flag{
			cli.StringSliceFlag{
				Name:  "unit",
				Usage: "Apply to paths that contain 'UNIT' test results",
			},
			cli.StringSliceFlag{
				Name:  "int",
				Usage: "Apply to paths that contain 'INTEGRATION' test results",
			},
			cli.StringSliceFlag{
				Name:  "e2e",
				Usage: "Apply to paths that contain 'E2E' (end-to-end) test results",
			},
			cli.StringFlag{
				Name:  "output",
				Usage: fmt.Sprintf("Path to generate '%s'  Must be a dir.  Defaults to working dir.", testReportName),
			},
		},
	}
	App.Commands = append(App.Commands, cmd)
}

func getTests(c *cli.Context) error {

	// Determine our verbosity.
	IsVerbose = c.Bool("verbose")

	// Grab our output path, we'll deal with it further down.
	testReportPath = c.String("output")

	testSuites, err := getJUnitXMLFiles(c)
	if err != nil {
		return cli.NewExitError(err, 0)
	}

	// Deal with our output path variable.
	if testReportPath != "" {
		if !strings.HasSuffix(testReportPath, "/") {
			testReportPath += "/"
		}
		if _, err := os.Stat(testReportPath); os.IsNotExist(err) {
			return cli.NewExitError(err, 0)
		}
		testReportName = testReportPath + testReportName
	}

	// Now iterate through our test suites from our files to calculate the totals for our new top-level object.
	totalTests := 0
	totalFails := 0
	totalErrors := 0
	totalTime := 0.0
	for i := 0; i < len(testSuites); i++ {
		testSuite := testSuites[i]
		totalTests += testSuite.numTests
		totalFails += testSuite.numFails
		totalErrors += testSuite.numErrors
		totalTime += testSuite.duration
	}

	doc := etree.NewDocument()
	doc.CreateProcInst("xml", `version="1.0" encoding="UTF-8"`)
	doc.CreateProcInst("xml-stylesheet", `type="text/xsl" href="style.xsl"`)

	docSuites := doc.CreateElement("testsuites")
	docSuites.CreateAttr("tests", fmt.Sprintf("%d", totalTests))
	docSuites.CreateAttr("fails", fmt.Sprintf("%d", totalFails))
	docSuites.CreateAttr("errors", fmt.Sprintf("%d", totalErrors))
	docSuites.CreateAttr("time", fmt.Sprintf("%f", totalTime))

	for i := 0; i < len(testSuites); i++ {
		fmt.Printf("[%s]: %s\n", testSuites[i].testType, testSuites[i].filePath)
		docSuites.AddChild(testSuites[i].rootElement)
	}

	doc.Indent(2)
	doc.WriteToFile(testReportName)
	fmt.Printf("------------------------------\n")
	fmt.Printf("Created: %s\n", testReportName)
	fmt.Printf("------------------------------\n")
	fmt.Printf("  tests:    %d\n", totalTests)
	fmt.Printf("  fails:    %d\n", totalFails)
	fmt.Printf("  errors:   %d\n", totalErrors)
	fmt.Printf("  duration: %.2fs\n", totalTime)
	fmt.Printf("------------------------------\n")

	return nil
}

func getTestTypeStr(testType string) string {
	if strings.Compare(testType, "int") == 0 {
		return "INTEGRATION"
	}
	if strings.Compare(testType, "e2e") == 0 {
		return "E2E"
	}
	return "UNIT"
}

func getValuesFromSuite(testSuite *etree.Element) (int, int, int, float64) {
	numTests, err := strconv.Atoi(testSuite.SelectAttrValue("tests", "0"))
	if err != nil {
		panic(err)
	}

	numFails, err := strconv.Atoi(testSuite.SelectAttrValue("failures", "0"))
	if err != nil {
		panic(err)
	}

	numErrors, err := strconv.Atoi(testSuite.SelectAttrValue("errors", "0"))
	if err != nil {
		panic(err)
	}

	duration, err := strconv.ParseFloat(testSuite.SelectAttrValue("time", "0.0"), 64)
	if err != nil {
		panic(err)
	}
	return numTests, numFails, numErrors, duration
}

func getJUnitXMLFiles(c *cli.Context) ([]JUnitXMLFile, error) {

	typeMap := map[string][]string{
		"unit": make([]string, 0),
		"int":  make([]string, 0),
		"e2e":  make([]string, 0),
	}

	// Handle our arguments that have no flag attached (or a flag that wasn't detected).
	detectedFlag := ""
	paths := c.Args()
	for i := 0; i < len(paths); i++ {
		path := paths.Get(i)

		// Due to a bug in cli, it's possible that a flagged entry ends up in this list.
		// Here, we check for this case and handle it accordingly adding the path to the appropriate type.
		if len(detectedFlag) != 0 {
			if detectedFlag == "output" {
				testReportPath = path
			} else {
				typeMap[detectedFlag] = append(typeMap[detectedFlag], path)
			}
			detectedFlag = ""
		} else if strings.HasPrefix(path, "--") {
			for testType := range typeMap {
				if strings.Compare(testType, path[2:]) == 0 {
					detectedFlag = testType
					break
				}
			}
			// If we aren't effected by this bug, then assume that the entry is of type `unit`.
		} else {
			typeMap["unit"] = append(typeMap["unit"], path)
		}
	}

	// Now handle our arguments that do have flags attached.
	for testType := range typeMap {
		typeMap[testType] = append(typeMap[testType], c.StringSlice(testType)...)
	}

	// Now that we've got all of the paths for our given type, we need to validate those paths and recurse through
	// directories if necessary.
	for testType, paths := range typeMap {
		for i := 0; i < len(paths); i++ {
			path := paths[i]
			additionalFiles := make([]string, 0)
			// Check to see if this argument is a valid path, if not then throw an error.
			pathStat, pathErr := os.Stat(path)
			if os.IsNotExist(pathErr) {
				return nil, pathErr
			}

			// If we have a directory, then we need to do some special stuff.
			if pathStat.IsDir() {
				var dirWalker = func(dirPath string, f os.FileInfo, err error) error {
					fileName := filepath.Base(dirPath)
					if strings.HasSuffix(fileName, ".xml") {
						additionalFiles = append(additionalFiles, dirPath)
					}
					return nil
				}
				filepath.Walk(path, dirWalker)
			}

			typeMap[testType] = append(typeMap[testType], additionalFiles...)
		}
	}

	// Finally, now that we've handled searching our directories, let's iterate once more to open our files and build our list of JUnitXMLFile structs.
	junitFiles := make([]JUnitXMLFile, 0)

	for testType, paths := range typeMap {
		for i := 0; i < len(paths); i++ {
			path := paths[i]
			xmlDoc := etree.NewDocument()
			//xmlDoc.ReadSettings.CharsetReader = charset.NewReaderLabel //TODO: Might need this for parsing special formats.
			if err := xmlDoc.ReadFromFile(path); err != nil {
				continue
			}
			// Check to see if our report is a collection of test suites.  If so, then we need to account for suites within.
			root := xmlDoc.SelectElement("testsuites")
			if root != nil {
				for _, testSuite := range root.SelectElements("testsuite") {
					numTests, numFails, numErrors, duration := getValuesFromSuite(testSuite)
					testSuite.CreateAttr("testType", getTestTypeStr(testType))
					junitFiles = append(junitFiles, JUnitXMLFile{
						rootElement: testSuite,
						filePath:    path,
						testType:    getTestTypeStr(testType),
						numTests:    numTests,
						numFails:    numFails,
						numErrors:   numErrors,
						duration:    duration,
					})
				}
			} else {
				testSuite := xmlDoc.SelectElement("testsuite")
				if testSuite == nil {
					continue
				}
				numTests, numFails, numErrors, duration := getValuesFromSuite(testSuite)
				testSuite.CreateAttr("testType", strings.ToUpper(testType))
				junitFiles = append(junitFiles, JUnitXMLFile{
					rootElement: testSuite,
					filePath:    path,
					testType:    getTestTypeStr(testType),
					numTests:    numTests,
					numFails:    numFails,
					numErrors:   numErrors,
					duration:    duration,
				})
			}
		}
	}

	return junitFiles, nil
}
